package com.atlassian.confluence.extra.flyingpdf.util;

import com.google.common.base.Preconditions;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * This is a representation of Confluence space which is going to be exported page-by-page.
 * It has table of contents and page tree with corresponding partial PDF files for each page.
 * Reads rendered PDF files and generates index (page title to location in PDF).
 */
public class ExportedSpaceStructure implements Serializable {

    private PdfNode tableOfContents;

    private final List<PdfNode> confluencePages;

    private final Map<String, Integer> locationByTitle = new HashMap<>();

    public ExportedSpaceStructure(PdfNode tableOfContents, List<PdfNode> confluencePages) {
        this.tableOfContents = tableOfContents;
        this.confluencePages = confluencePages;

        final AtomicInteger currentPage = new AtomicInteger(tableOfContents.getRenderedPdfFile().getNumPages() + 1);
        confluencePages.forEach(node -> initialise(node, currentPage));
    }

    /**
     * Change TOC for this space. We might want to do this, because TOC generation is done in 2 steps.
     */
    public void replaceToc(PdfNode newTableOfContents) {
        Preconditions.checkArgument(
                tableOfContents.getRenderedPdfFile().getNumPages()
                        == newTableOfContents.getRenderedPdfFile().getNumPages(),
                "TOC should be replaced only with TOC of the same size. " +
                        "It is important because TOC size will affect TOC content (page offsets). " +
                        "This is a bit of chicken and egg problem, which makes us generate TOC twice.");

        tableOfContents = newTableOfContents;
    }

    /**
     * Calculate locations for each page
     */
    private void initialise(PdfNode node, AtomicInteger currentPage) {
        locationByTitle.put(node.getPageTitle(), currentPage.get());
        currentPage.addAndGet(node.getRenderedPdfFile().getNumPages());

        for (PdfNode child : node.getChildren()) {
            initialise(child, currentPage);
        }
    }

    public PdfNode getTableOfContents() {
        return tableOfContents;
    }

    public List<PdfNode> getConfluencePages() {
        return Collections.unmodifiableList(confluencePages);
    }

    Integer locationByNode(PdfNode node) {
        if (tableOfContents.equals(node)) {
            return 1;
        }

        return locationByTitle.get(node.getPageTitle());
    }

    public Integer locationByTitle(String title) {
        return locationByTitle.get(title);
    }

    public boolean hasPageTitle(String title) {
        return locationByTitle.containsKey(title);
    }

    public Map<String, Integer> locationByTitleMap() {
        return Collections.unmodifiableMap(locationByTitle);
    }
}
