package com.atlassian.confluence.extra.flyingpdf.sandbox;

import com.atlassian.confluence.extra.flyingpdf.html.DecorationPolicy;
import com.atlassian.confluence.extra.flyingpdf.util.ExportedSpaceStructure;
import com.atlassian.confluence.util.sandbox.SandboxSerializer;
import org.apache.commons.lang3.SerializationUtils;

import java.util.ArrayList;
import java.util.List;

import static com.atlassian.confluence.util.sandbox.SandboxSerializers.compositeByteArraySerializer;
import static com.atlassian.confluence.util.sandbox.SandboxSerializers.stringSerializer;

public class SandboxPdfJoinRequest {
    private final String spaceKey;
    private final ExportedSpaceStructure exportedSpaceStructure;
    private final String outputFile;
    private final String baseUrl;
    private final DecorationPolicy decorationPolicy;

    SandboxPdfJoinRequest(String spaceKey, ExportedSpaceStructure exportedSpaceStructure,
                          String outputFile, String baseUrl, DecorationPolicy decorationPolicy) {
        this.spaceKey = spaceKey;
        this.exportedSpaceStructure = exportedSpaceStructure;
        this.outputFile = outputFile;
        this.baseUrl = baseUrl;
        this.decorationPolicy = decorationPolicy;
    }

    static Serializer serializer() {
        return Serializer.instance;
    }

    public String getSpaceKey() {
        return spaceKey;
    }

    ExportedSpaceStructure getExportedSpaceStructure() {
        return exportedSpaceStructure;
    }

    String getOutputFile() {
        return outputFile;
    }

    public String getBaseUrl() {
        return baseUrl;
    }

    public DecorationPolicy getDecorationPolicy() {
        return decorationPolicy;
    }

    static final class Serializer implements SandboxSerializer<SandboxPdfJoinRequest> {
        static final Serializer instance = new Serializer();

        private Serializer() {
        }

        @Override
        public byte[] serialize(SandboxPdfJoinRequest request) {
            List<byte[]> fields = new ArrayList<>();

            fields.add(stringSerializer().serialize(request.getSpaceKey()));
            fields.add(structureToBytes(request.getExportedSpaceStructure()));
            fields.add(stringSerializer().serialize(request.getOutputFile()));
            fields.add(stringSerializer().serialize(request.getBaseUrl()));
            fields.add(decorationPolicyToBytes(request.getDecorationPolicy()));

            return compositeByteArraySerializer().serialize(fields);
        }

        @Override
        public SandboxPdfJoinRequest deserialize(byte[] bytes) {
            List<byte[]> fields = compositeByteArraySerializer().deserialize(bytes);
            final String spaceKey = stringSerializer().deserialize(fields.get(0));
            final ExportedSpaceStructure exportedSpaceStructure = structureFromBytes(fields.get(1));
            final String outputFile = stringSerializer().deserialize(fields.get(2));
            final String baseUrl = stringSerializer().deserialize(fields.get(3));
            final DecorationPolicy decorationPolicy = decorationPolicyFromBytes(fields.get(4));

            return new SandboxPdfJoinRequest(
                    spaceKey,
                    exportedSpaceStructure,
                    outputFile,
                    baseUrl,
                    decorationPolicy);
        }

        private static byte[] structureToBytes(ExportedSpaceStructure exportedSpaceStructure) {
            return SerializationUtils.serialize(exportedSpaceStructure);
        }

        private static ExportedSpaceStructure structureFromBytes(byte[] bytes) {
            return (ExportedSpaceStructure) SerializationUtils.deserialize(bytes);
        }

        private static byte[] decorationPolicyToBytes(DecorationPolicy decorationPolicy) {
            return SerializationUtils.serialize(decorationPolicy);
        }

        private static DecorationPolicy decorationPolicyFromBytes(byte[] bytes) {
            return (DecorationPolicy) SerializationUtils.deserialize(bytes);
        }
    }
}
