package com.atlassian.confluence.extra.flyingpdf.html;

import org.apache.log4j.Logger;
import org.jsoup.Jsoup;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xhtmlrenderer.css.constants.CSSName;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.io.StringWriter;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;

public class AutoFontScaleUtils {
    private static final double FULL_SIZE = 1d;

    private static final int MIN_COLUMN_TO_APPLY_CHARACTER_COUNTING = 5;
    private static final int EMPTY = 0;

    private static final String ELEMENT_TABLE = "table";
    private static final String ELEMENT_TR = "tr";
    private static final String ELEMENT_TD = "td";
    private static final String ELEMENT_TH = "th";
    private static final String ELEMENT_IMG = "img";

    private static final String ATTRIBUTE_CLASS = "class";
    private static final String ATTRIBUTE_STYLE = "style";

    private static final String CSS_CLASS_NAME_CONFLUENCE_TABLE = "confluenceTable";

    private static final Logger LOG = Logger.getLogger(AutoFontScaleUtils.class);


    /**
     * Scale down all the table elements in the document that has confluenceTable class based on
     * its number of columns and its characters count
     */
    public static void applyTableScalingLogic(Document xhtmlDocument) {
        if (xhtmlDocument == null) {
            return;
        }
        long start = System.currentTimeMillis();
        NodeList tables = xhtmlDocument.getElementsByTagName(ELEMENT_TABLE);
        for (int i = 0; i < tables.getLength(); i++) {
            Node table = tables.item(i);
            if (table.getNodeType() == Node.ELEMENT_NODE) {
                Element tableElement = (Element) table;
                if (tableElement.getAttribute(ATTRIBUTE_CLASS).contains(CSS_CLASS_NAME_CONFLUENCE_TABLE)) {
                    scaleTableBaseOnNumberOfColumnsAndCharacter(tableElement);
                }
            }
        }
        LOG.debug("Total scaling time : " + (System.currentTimeMillis() - start));
    }

    /**
     * Scale down the table element based on
     * its number of columns and its characters count
     *
     * @param tableElement html table element to be scaled down
     */
    public static void scaleTableBaseOnNumberOfColumnsAndCharacter(Element tableElement) {
        if (tableElement == null) {
            return;
        }
        long time = System.currentTimeMillis();
        int numOfColumn = detectNumberOfColumnsFromElement(tableElement);
        double colScaleFontSize = FULL_SIZE;
        if (numOfColumn > 0) {
            colScaleFontSize = FontRangeHelper.getColumnCountInstance().getFontSize(numOfColumn).doubleValue();
        }
        LOG.debug("column count processing time : " + (System.currentTimeMillis() - time));

        time = System.currentTimeMillis();
        int noOfTRCharacters = detectCrossTableMaxCharacters(tableElement);
        double chaScaleFontSize = FULL_SIZE;
        if (numOfColumn >= MIN_COLUMN_TO_APPLY_CHARACTER_COUNTING) {
            chaScaleFontSize = FontRangeHelper.getCharacterCountInstance().getFontSize(noOfTRCharacters).doubleValue();
        }
        LOG.debug("character count processing time : " + (System.currentTimeMillis() - time));

        double finalRatio = Math.min(colScaleFontSize, chaScaleFontSize);
        setStyleToElement(tableElement, CSSName.FONT_SIZE.toString(), String.valueOf(finalRatio) + "em");

        adjustTablePadding(tableElement, finalRatio);
        LOG.debug("noOfTRCharacters " + noOfTRCharacters);
        LOG.debug("numOfColumn " + numOfColumn);
        LOG.debug("colScaleFontSize " + colScaleFontSize);
        LOG.debug("chaScaleFontSize " + chaScaleFontSize);
        LOG.debug("final " + Math.min(colScaleFontSize, chaScaleFontSize));
    }

    /**
     * Scale down the table element based on its number of columns
     *
     * @param tableElement html table element to be scaled down
     */
    public static void scaleTableBaseOnNumberOfColumns(Element tableElement) {
        if (tableElement == null) {
            return;
        }
        int numOfColumn = detectNumberOfColumnsFromElement(tableElement);
        if (numOfColumn == 0) {
            return;
        }
        double scaleRatio = FontRangeHelper.getColumnCountInstance().getFontSize(numOfColumn).doubleValue();
        setStyleToElement(tableElement, CSSName.FONT_SIZE.toString(), String.valueOf(scaleRatio) + "em");
    }

    /**
     * Scale down the table element based on its number of characters
     *
     * @param tableElement html table element to be scaled down
     */
    public static void scaleTableBaseOnNumberOfCharacters(Element tableElement) {
        if (tableElement == null) {
            return;
        }
        int noOfTRCharacters = detectCrossTableMaxCharacters(tableElement);
        double scaleRatio = FontRangeHelper.getCharacterCountInstance().getFontSize(noOfTRCharacters).doubleValue();
        setStyleToElement(tableElement, CSSName.FONT_SIZE.toString(), String.valueOf(scaleRatio) + "em");
    }

    /**
     * Get string content of a Node element
     *
     * @param doc Node to get content from
     * @return String content of the provided node
     */
    public static String getStringFromDocument(Node doc) {
        try {
            DOMSource domSource = new DOMSource(doc);
            StringWriter writer = new StringWriter();
            StreamResult result = new StreamResult(writer);
            TransformerFactory tf = TransformerFactory.newInstance();
            Transformer transformer = tf.newTransformer();
            transformer.transform(domSource, result);
            return writer.toString();
        } catch (TransformerException ex) {
            LOG.debug(ex);
            return "";
        }
    }

    /*-------------- private things --------------*/

    private static void adjustTablePadding(Element tableElement, double finalRatio) {
        Collection<Element> tds = findElementsByTagName(tableElement, ELEMENT_TD);
        for (Element tdElement : tds) {
            setStyleToElement(tdElement, CSSName.PADDING_SHORTHAND.toString(), finalRatio / 2 + "em");
        }
    }

    private static Collection<Element> findElementsByTagName(Element parentElement, String tagName) {
        NodeList nodeList = parentElement.getElementsByTagName(tagName);

        if (nodeList.getLength() == EMPTY) return Collections.emptySet();

        Collection<Element> elements = new HashSet<Element>(nodeList.getLength());
        for (int i = 0; i < nodeList.getLength(); i++) {
            Node node = nodeList.item(i);
            if (node.getNodeType() == Node.ELEMENT_NODE) {
                elements.add((Element) node);
            }
        }
        return elements;
    }

    private static void setStyleToElement(Element e, String styleName, String styleValue) {
        String currentStyle = e.getAttribute(ATTRIBUTE_STYLE);
        e.setAttribute(ATTRIBUTE_STYLE, currentStyle + "; " + styleName + ":" + styleValue);
    }

    private static int detectCrossTableMaxCharacters(Element tableElement) {
        NodeList rows = tableElement.getElementsByTagName(ELEMENT_TR);
        if (rows.getLength() == EMPTY) {
            LOG.info("Processing table if empty row");
            return EMPTY;
        }

        Element firstRow = ((Element) rows.item(0));
        NodeList columns = firstRow.getElementsByTagName(ELEMENT_TH);
        if (columns.getLength() == EMPTY) {
            // try td
            columns = firstRow.getElementsByTagName(ELEMENT_TD);
        }

        Integer[] largestCellByColumns = calculateLargestCellByColumns(rows, columns);

        int countAggregatedLargestCell = 0;
        for (int i = 0; i < largestCellByColumns.length; i++) {
            countAggregatedLargestCell += largestCellByColumns[i];
        }
        return countAggregatedLargestCell;
    }

    private static Integer[] calculateLargestCellByColumns(NodeList rows, NodeList columns) {
        Integer[] largestCellByColumns = new Integer[columns.getLength()];
        // init array value with all 0 count
        for (int i = 0; i < largestCellByColumns.length; i++) {
            largestCellByColumns[i] = 0;
        }

        // process on table matrix
        for (int tr = 0; tr < rows.getLength(); tr++) {
            Element trElement = (Element) rows.item(tr);
            NodeList tds = trElement.getElementsByTagName(ELEMENT_TD);
            if (tds.getLength() == EMPTY) {
                tds = trElement.getElementsByTagName(ELEMENT_TH);
            }
            for (int td = 0; td < columns.getLength(); td++) {
                Element tdElement = (Element) tds.item(td);
                String content = getStringFromDocument(tdElement);
                // now strip all HTML tags
                content = Jsoup.parseBodyFragment(content).text();
                if (content.length() > largestCellByColumns[td]) {
                    largestCellByColumns[td] = content.length();
                    LOG.debug(content.length() + " " + content + "\n");
                }
            }
        }
        return largestCellByColumns;
    }

    private static int detectNumberOfColumnsFromElement(Element tableElement) {
        // try thead
        NodeList trs = tableElement.getElementsByTagName(ELEMENT_TR);
        if (trs.getLength() == 0) {
            return EMPTY;
        }
        int noOfColumns = 0;
        for (int i = 0; i < trs.getLength(); i++) {
            int columnsPerRow = ((Element) trs.item(i)).getChildNodes().getLength();
            if (columnsPerRow > noOfColumns) {
                noOfColumns = columnsPerRow;
            }
        }
        return noOfColumns;
    }

    private static class FontRangeHelper<E extends Number> {

        private Map<Integer[], E> internalRangeMap = new HashMap<Integer[], E>();

        private static FontRangeHelper<Double> columnCountInstance = null;
        private static FontRangeHelper<Double> characterCountInstance = null;

        static {
            initialize();
        }

        private static void initialize() {
            getColumnCountInstance();
            getCharacterCountInstance();
        }

        static FontRangeHelper<Double> getColumnCountInstance() {
            if (null == columnCountInstance) {
                columnCountInstance = new FontRangeHelper<Double>();
                columnCountInstance
                        .setRange(1, 3, 1d)
                        .setRange(4, 7, 0.9)
                        .setRange(8, 12, 0.8)
                        .setRange(13, Integer.MAX_VALUE, 0.7);
            }
            return columnCountInstance;
        }

        static FontRangeHelper<Double> getCharacterCountInstance() {
            if (null == characterCountInstance) {
                characterCountInstance = new FontRangeHelper<Double>();
                characterCountInstance
                        .setRange(1, 600, 1d)
                        .setRange(601, 1000, 0.9)
                        .setRange(1001, 2000, 0.8)
                        .setRange(2001, Integer.MAX_VALUE, 0.7);
            }
            return characterCountInstance;
        }

        private FontRangeHelper() {
        }

        private FontRangeHelper<E> setRange(int start, int end, E d) {
            Integer[] range = new Integer[2];
            range[0] = start;
            range[1] = end;
            this.internalRangeMap.put(range, d);
            return this;
        }

        Number getFontSize(int numOfColumn) {
            for (Entry<Integer[], E> entry : this.internalRangeMap.entrySet()) {
                Integer[] range = entry.getKey();
                if (numOfColumn >= range[0] && numOfColumn <= range[1]) {
                    return entry.getValue();
                }
            }
            return FULL_SIZE;
        }
    }

}
