package com.atlassian.confluence.extra.flyingpdf.analytic;

import com.atlassian.analytics.api.annotations.EventName;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;

import java.util.Collections;

import static java.nio.charset.Charset.defaultCharset;

/**
 * Analytic event about PDF export: page or space
 */
@EventName("confluence.pdf.export.run")
public class PdfExportAnalyticEvent {

    private final EnvironmentInfo environmentInfo;

    private final ExportResults exportResults;

    private final int confluencePagesTotal;

    private final int confluencePagesProcessed;

    private final long failedPageIdHash;

    private final long failedPageRevisionHash;

    private final TimeStatistics timeStatistics;

    private final int totalTime;

    private final int tocBuildTime;

    private final int joinTime;

    @SuppressWarnings("UnstableApiUsage")
    private static final HashFunction numbersHash = Hashing.sha256();

    @SuppressWarnings("UnstableApiUsage")
    private static final HashFunction stringsHash = Hashing.sha256();

    public PdfExportAnalyticEvent(PageExportMetrics pageExportMetrics) {
        this.environmentInfo = pageExportMetrics.getEnvironmentInfo();
        this.exportResults = pageExportMetrics.getExportResults();
        this.confluencePagesTotal = 1;
        // There was a fail on the page, but we still partially processed it
        // This is consistent with space information which includes a page that caused failure in 'confluencePagesProcessed'
        this.confluencePagesProcessed = 1;

        if (ExportStatus.isFail(exportResults.getExportStatus())) {
            this.failedPageIdHash = numbersHash.hashLong(pageExportMetrics.getPageId()).asLong();
            this.failedPageRevisionHash = numbersHash.hashLong(pageExportMetrics.getPageRevision()).asLong();
        } else {
            this.failedPageIdHash = -1;
            this.failedPageRevisionHash = -1;
        }

        this.timeStatistics = new TimeStatistics(Collections.singleton(pageExportMetrics));
        // Since we export single page, it is safe to do
        this.totalTime = timeStatistics.getMaxPageTime();
        this.tocBuildTime = -1;
        this.joinTime = -1;
    }

    public PdfExportAnalyticEvent(SpaceExportMetrics spaceExportMetrics) {
        final PageExportMetrics failedPage = spaceExportMetrics.getPageExportMetrics().stream()
                .filter(p -> ExportStatus.isFail(p.getExportResults().getExportStatus()))
                .findAny()
                .orElse(null);

        if (failedPage != null) {
            spaceExportMetrics.getExportResults().setExportStatus(failedPage.getExportResults().getExportStatus());
            spaceExportMetrics.getExportResults().setFailureLocation(FailureLocation.PAGE);

            this.failedPageIdHash = numbersHash.hashLong(failedPage.getPageId()).asLong();
            this.failedPageRevisionHash = numbersHash.hashLong(failedPage.getPageRevision()).asLong();
        } else {
            this.failedPageIdHash = -1;
            this.failedPageRevisionHash = -1;
        }

        this.environmentInfo = spaceExportMetrics.getEnvironmentInfo();
        this.exportResults = spaceExportMetrics.getExportResults();
        this.confluencePagesTotal = spaceExportMetrics.getConfluencePages();
        if (spaceExportMetrics.getEnvironmentInfo().getSandboxStatus() == SandboxStatus.USED) {
            // If we exported with sandbox, we know how many pages we processed
            this.confluencePagesProcessed = spaceExportMetrics.getPageExportMetrics().size();
        } else if (ExportStatus.isSuccessful(spaceExportMetrics.getExportResults().getExportStatus())) {
            // If sandbox was not used and the export was successful, we processed all pages
            this.confluencePagesProcessed = spaceExportMetrics.getConfluencePages();
        } else {
            // Otherwise sandbox wasn't used and we failed. We don't know how many pages were processed OK.
            this.confluencePagesProcessed = -1;
        }
        this.timeStatistics = new TimeStatistics(spaceExportMetrics.getPageExportMetrics());
        this.totalTime = spaceExportMetrics.getTotalTime();
        this.tocBuildTime = spaceExportMetrics.getTocBuildTime();
        this.joinTime = spaceExportMetrics.getJoinTime();
    }

    public int getDcNodeId() {
        return environmentInfo.getDcNodeId();
    }

    public SandboxStatus getSandboxStatus() {
        return environmentInfo.getSandboxStatus();
    }

    public ExportScope getExportScope() {
        return environmentInfo.getExportScope();
    }

    public long getSpaceKeyHash() {
        return stringsHash.hashString(environmentInfo.getSpaceKey(), defaultCharset()).asLong();
    }

    public int getTotalTime() {
        return totalTime;
    }

    public int getMinPageTime() {
        return timeStatistics.getMinPageTime();
    }

    public int getMaxPageTime() {
        return timeStatistics.getMaxPageTime();
    }

    public int getMeanPageTime() {
        return timeStatistics.getMeanPageTime();
    }

    public int getP50PageTime() {
        return timeStatistics.getP50PageTime();
    }

    public int getP95PageTime() {
        return timeStatistics.getP95PageTime();
    }

    public int getP98PageTime() {
        return timeStatistics.getP98PageTime();
    }

    public int getP99PageTime() {
        return timeStatistics.getP99PageTime();
    }

    public int getP999PageTime() {
        return timeStatistics.getP999PageTime();
    }

    public int getTocBuildTime() {
        return tocBuildTime;
    }

    public int getJoinTime() {
        return joinTime;
    }

    public ExportStatus getExportStatus() {
        return exportResults.getExportStatus();
    }

    public int getConfluencePagesTotal() {
        return confluencePagesTotal;
    }

    public int getConfluencePagesProcessed() {
        return confluencePagesProcessed;
    }

    public int getPdfPagesCount() {
        return exportResults.getPdfPagesTotal();
    }

    public long getPdfFileSizeBytes() {
        return exportResults.getPdfFileSizeBytes();
    }

    public FailureLocation getFailureLocation() {
        return exportResults.getFailureLocation();
    }

    public long getFailedPageIdHash() {
        return failedPageIdHash;
    }

    public long getFailedPageRevisionHash() {
        return failedPageRevisionHash;
    }
}
