package com.atlassian.confluence.compat.setup.xstream;

import aQute.bnd.annotation.component.Component;
import com.atlassian.confluence.api.service.exceptions.ServiceException;
import com.atlassian.confluence.setup.xstream.XStreamManager;
import com.atlassian.spring.container.ContainerManager;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.converters.Converter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Reader;
import java.io.Writer;
import java.lang.reflect.Method;

/**
 * This class lets plugins to be compatible with all supported versions of Confluence for XStream.
 * Since 7.10 Confluence uses newer XStream and has added methods in XStreamManager.
 * XStreamManager is responsible for choosing between XStreamManager or creating a new XStream(Confluence 7.9 and prior)
 */
@Component
public class XStreamManagerCompat {
    private static final Logger log = LoggerFactory.getLogger(XStreamManagerCompat.class);
    private final Supplier<XStreamCompat> delegate;

    public XStreamManagerCompat() {
        this((XStreamManager) ContainerManager.getComponent("xStreamManager"),
                XStreamManagerCompat.class.getClassLoader());
    }

    public XStreamManagerCompat(XStreamManager xStreamManager, ClassLoader classLoader) {
        delegate = Suppliers.memoize(() -> initialiseXStreamCompat(xStreamManager, classLoader));
    }

    private XStreamCompat initialiseXStreamCompat(XStreamManager xStreamManager, ClassLoader classLoader) {
        XStreamCompat internalDelegate;
        try {
            Object pluginXStream = getConfluenceXStream(xStreamManager, classLoader);
            internalDelegate = new ConfluenceXStreamCompat(pluginXStream);
        } catch (ClassNotFoundException e) {
            log.debug("Could not find Confluence XStream, falling back to Confluence 7.9 or prior XStream impl.", e);
            XStream xStream = new XStream();
            xStream.setClassLoader(classLoader);
            internalDelegate = new XStream111Compat(xStream);
        } catch (ReflectiveOperationException e) {
            throw new ServiceException("Confluence XStream couldn't be initialized.", e);
        }
        return internalDelegate;
    }

    /**
     * Serialize an object to a pretty-printed XML String.
     */
    public String toXML(Object obj) {
        return delegate.get().toXML(obj);
    }

    /**
     * Serialize an object to the given Writer as pretty-printed XML. The Writer will be flushed afterwards and in case
     * of an exception.
     */
    public void toXML(Object obj, Writer writer) {
        delegate.get().toXML(obj, writer);
    }

    /**
     * Deserialize an object from an XML String.
     */
    public Object fromXML(String xml) {
        return delegate.get().fromXML(xml);
    }

    /**
     * Deserialize an object from an XML Reader.
     */
    public Object fromXML(Reader reader){
        return delegate.get().fromXML(reader);
    }

    /**
     * Provides a way to access 1.1.1 XStream object
     * Supposed to be removed when Confluence 7.9 is EOL.
     *
     * @return XStream 1.1.1 equivalent object.
     *
     */
    public XStream getXStream() {
        return delegate.get().getXStream();
    }

    /**
     * Register a converter with the appropriate priority.
     *
     * @param converter XStream converter to register
     * @param priority Priority to be used in XStream
     */
    public void registerConverter(Converter converter, Integer priority) {
        delegate.get().registerConverter(converter, priority);
    }

    /**
     * Alias a Class to a shorter name to be used in XML elements.
     *
     * @param name Short name
     * @param type  Type to be aliased
     */
    public void alias(String name, Class<?> type) {
        delegate.get().alias(name, type);
    }

    private Object getConfluenceXStream(XStreamManager xStreamManager, ClassLoader classLoader) throws ReflectiveOperationException {
        Method getPluginXStreamMethod =
        Class.forName("com.atlassian.confluence.setup.xstream.ConfluenceXStreamManager", true, classLoader)
                .getMethod("getPluginXStream", ClassLoader.class);
        return getPluginXStreamMethod.invoke(xStreamManager, classLoader);
    }
}
