package com.atlassian.confluence.compat.setup.xstream;

import com.atlassian.confluence.api.service.exceptions.ServiceException;
import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.converters.Converter;

import java.io.Reader;
import java.io.Writer;
import java.lang.reflect.Method;

/**
 * Internal XStream compatibility class for Confluence 7.10 and above
 */
class ConfluenceXStreamCompat implements XStreamCompat {
    private static final String COULD_NOT_DESERIALIZE_OBJECT =
            "Could not deserialize object as XStream might not be properly initialized";
    private static final String COULD_NOT_SERIALIZE_OBJECT =
            "Could not serialize object as XStream might not be properly initialized";

    private final Object confluenceXStream;
    private final Method toXMLObjMethod;
    private final Method toXMLObjOnWriterMethod;
    private final Method fromXMLMethod;
    private final Method fromXMLReaderMethod;
    private final Method getXStreamMethod;
    private final Method registerConverterMethod;
    private final Method aliasMethod;

    ConfluenceXStreamCompat(Object confluenceXStream) throws ReflectiveOperationException {
        this.confluenceXStream = confluenceXStream;
        toXMLObjMethod = getConfluenceXStreamMethod("toXML", Object.class);
        toXMLObjOnWriterMethod = getConfluenceXStreamMethod("toXML", Object.class, Writer.class);
        fromXMLMethod = getConfluenceXStreamMethod("fromXML", String.class);
        fromXMLReaderMethod = getConfluenceXStreamMethod("fromXML", Reader.class);
        getXStreamMethod = getConfluenceXStreamMethod("getXStream");
        registerConverterMethod = getConfluenceXStreamMethod("registerConverter", Converter.class, Integer.class);
        aliasMethod = getConfluenceXStreamMethod("alias", String.class, Class.class);
    }
    @Override
    public String toXML(Object obj) {
        try {
            return (String) toXMLObjMethod.invoke(confluenceXStream, obj);
        } catch (ReflectiveOperationException e) {
            throw new ServiceException(COULD_NOT_SERIALIZE_OBJECT, e);
        }
    }

    @Override
    public void toXML(Object obj, Writer writer) {
        try {
            toXMLObjOnWriterMethod.invoke(confluenceXStream, obj, writer);
        } catch (ReflectiveOperationException e) {
            throw new ServiceException(COULD_NOT_SERIALIZE_OBJECT, e);
        }
    }

    @Override
    public Object fromXML(String xml) {
        try {
            return fromXMLMethod.invoke(confluenceXStream, xml);
        } catch (ReflectiveOperationException e) {
            throw new ServiceException(COULD_NOT_DESERIALIZE_OBJECT, e);
        }
    }

    @Override
    public Object fromXML(Reader reader) {
        try {
            return fromXMLReaderMethod.invoke(confluenceXStream, reader);
        } catch (ReflectiveOperationException e) {
            throw new ServiceException(COULD_NOT_DESERIALIZE_OBJECT, e);
        }
    }

    @Override
    public XStream getXStream() {
        try {
            return (XStream) getXStreamMethod.invoke(confluenceXStream);
        } catch (ReflectiveOperationException e) {
            throw new ServiceException(COULD_NOT_DESERIALIZE_OBJECT, e);
        }
    }

    @Override
    public void registerConverter(Converter converter, Integer priority) {
        try {
            registerConverterMethod.invoke(confluenceXStream, converter, priority);
        } catch (ReflectiveOperationException e) {
            throw new ServiceException(COULD_NOT_DESERIALIZE_OBJECT, e);
        }
    }

    @Override
    public void alias(String name, Class<?> type) {
        try {
            aliasMethod.invoke(confluenceXStream, name, type);
        } catch (ReflectiveOperationException e) {
            throw new ServiceException(COULD_NOT_DESERIALIZE_OBJECT, e);
        }
    }

    private Method getConfluenceXStreamMethod(String methodName, Class<?>... parameterTypes) throws ReflectiveOperationException {
        return Class.forName("com.atlassian.confluence.setup.xstream.ConfluenceXStreamInternal")
                .getMethod(methodName, parameterTypes);
    }
}
