package com.atlassian.config.lifecycle;

import com.atlassian.plugin.PluginManager;
import com.atlassian.johnson.event.Event;
import com.atlassian.johnson.event.EventType;
import com.atlassian.johnson.event.EventLevel;
import com.atlassian.event.EventManager;
import com.atlassian.config.lifecycle.events.ApplicationStoppedEvent;
import com.atlassian.config.lifecycle.events.ApplicationStartedEvent;
import com.atlassian.config.lifecycle.events.ApplicationStoppingEvent;

import javax.servlet.ServletContext;
import java.util.List;
import java.util.Collections;
import java.util.Iterator;

import org.apache.log4j.Logger;

public class DefaultLifecycleManager implements LifecycleManager
{
    private static Logger log = Logger.getLogger(LifecycleManager.class);

    private PluginManager pluginManager;
    private EventManager eventManager;

    public void startUp(ServletContext servletContext)
    {
        List moduleDescriptors = getLifecyclePluginModuleDescriptors();
        LifecycleContext context = new DefaultLifecycleContext(servletContext);

        LifecyclePluginModuleDescriptor descriptor = null;

        try
        {
            for (Iterator it = moduleDescriptors.iterator(); it.hasNext();)
            {
                descriptor = (LifecyclePluginModuleDescriptor) it.next();
                LifecycleItem item = (LifecycleItem) descriptor.getModule();
                log.info("Starting: " + descriptor);
                item.startup(context);
            }

            eventManager.publishEvent(new ApplicationStartedEvent(this));
        }
        catch (Throwable t)
        {
            panicAndShutdown(t, context, descriptor);
        }
    }

    public void shutDown(ServletContext servletContext)
    {
        shutDown(servletContext, null);
    }

    private void panicAndShutdown(Throwable t, LifecycleContext context, LifecyclePluginModuleDescriptor descriptor)
    {
        String errorString = "Unable to start up Confluence. Fatal error during startup sequence: " + descriptor + " - " + t;
        log.fatal(errorString, t);
        context.getAgentJohnson().addEvent(new Event(EventType.get("startup"), errorString, EventLevel.FATAL));
        shutDown(context.getServletContext(), descriptor.getCompleteKey());
    }

    private void shutDown(ServletContext servletContext, String startingPluginKey)
    {
        eventManager.publishEvent(new ApplicationStoppingEvent(this));

        List moduleDescriptors = getLifecyclePluginModuleDescriptors();
        Collections.reverse(moduleDescriptors);
        LifecycleContext context = new DefaultLifecycleContext(servletContext);

        boolean started = startingPluginKey == null;

        for (Iterator it = moduleDescriptors.iterator(); it.hasNext();)
        {
            LifecyclePluginModuleDescriptor descriptor = (LifecyclePluginModuleDescriptor) it.next();

            if (!started)
            {
                if (descriptor.getCompleteKey().equals(startingPluginKey))
                {
                    started = true;
                }
                else
                {
                    continue;
                }
            }

            log.info("Shutting down: " + descriptor);
            LifecycleItem item = (LifecycleItem) descriptor.getModule();
            try
            {
                item.shutdown(context);
            }
            catch (Throwable t)
            {
                log.error("Error running shutdown plugin: " + descriptor.getDescription() + " - " + t, t);
            }
        }

        eventManager.publishEvent(new ApplicationStoppedEvent(this));
    }

    private List getLifecyclePluginModuleDescriptors()
    {
        List modules = pluginManager.getEnabledModuleDescriptorsByClass(LifecyclePluginModuleDescriptor.class);
        Collections.sort(modules);
        return modules;
    }

    public void setPluginManager(PluginManager pluginManager)
    {
        this.pluginManager = pluginManager;
    }

    public void setEventManager(EventManager eventManager)
    {
        this.eventManager = eventManager;
    }
}
