package com.atlassian.ccev;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.ThreadSafe;
import javax.inject.Named;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Validates emails and checks duplicates in a way compatible with Jira Cloud validation
 */
@ParametersAreNonnullByDefault
@ThreadSafe
@Named
public class EmailValidator {
    /**
     * Check for invalid and duplicated emails in given list
     *
     * @param emailProviders list of emails
     * @return List of invalid emails. The order of returned list is not the same as input list.
     */
    public <T> List<Result<T>> validate(Iterable<T> emailProviders, Function<T, String> getEmail) {
        Map<String, List<Result<T>>> processedEmails = new HashMap<>();
        emailProviders.forEach(ep -> validate(ep, getEmail, processedEmails));
        return processedEmails.values().stream()
                .flatMap(Collection::stream)
                .filter(this::isInvalidOrDuplicated)
                .collect(Collectors.toList());
    }

    /**
     * Check if email is valid. Does not check for duplicated emails.
     * @return Whether the email is valid. Null and empty emails are invalid.
     */
    public boolean validate(@Nullable String email) {
        return email != null && AidEmailAddress.validEmail(email) && ValidatorJs.isEmail(email);
    }

    private <T> boolean isInvalidOrDuplicated(Result<T> result) {
        return result.isDuplicated() || !result.isValid() || result.isEmpty();
    }

    private <T> void validate(T user, Function<T, String> getEmail, Map<String, List<Result<T>>> processedEmails) {
        String email = getNonNullEmail(user, getEmail);

        processedEmails.compute(email.toLowerCase(), (key, results) -> {
            if (results == null) {
                results = new ArrayList<>();
            } else {
                results.get(0).duplicated = true;
            }
            results.add(new Result<>(user,
                    validate(email),
                    email.isEmpty(),
                    !results.isEmpty()));
            return results;
        });
    }

    private <T> String getNonNullEmail(T user, Function<T, String> getEmail) {
        final String email = getEmail.apply(user);
        return email != null ? email : "";
    }

    public static class Result<T> {
        private final T user;
        private final boolean valid;
        private final boolean empty;
        private boolean duplicated;

        public Result(final T user, final boolean valid, final boolean empty, final boolean duplicated) {
            this.user = user;
            this.valid = valid;
            this.empty = empty;
            this.duplicated = duplicated;
        }

        public T getUser() {
            return user;
        }

        public boolean isValid() {
            return valid;
        }

        public boolean isEmpty() {
            return empty;
        }

        public boolean isDuplicated() {
            return duplicated;
        }
    }
}
