package com.atlassian.ccev;

import javax.annotation.ParametersAreNonnullByDefault;
import java.net.IDN;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.function.BooleanSupplier;
import java.util.function.Predicate;
import java.util.regex.Pattern;

/**
 * Imitates Atlassian ID email validation
 */
@ParametersAreNonnullByDefault
class AidEmailAddress {
    /**
     * Check if email is valid
     *
     * https://stash.atlassian.com/projects/aid/repos/aid-libraries/browse/core/src/main/scala/com/atlassian/id/user/package.scala?at=a7dcf0e470efb657adc84d5e030213fd1f457e4f#55-111
     *
     * @return whether email is valid
     */
    static boolean validEmail(final String emailString) {
        final String trimmedEmailString = emailString.trim();
        final String[] parts = trimmedEmailString.split("@");

        if (parts.length != 2) { // equivalent of containsAtSign in AID code
            return false;
        }

        final Predicate<String> nonEmpty = trimmedEmail -> !trimmedEmail.isEmpty();
        final Predicate<String> usernameMaxLength = userName -> userName.getBytes(StandardCharsets.UTF_8).length <= 80;
        final Predicate<String> validDomain = domain -> {
            final Predicate<String> hasMultipleParts = dm -> dm.matches(".+\\..+");
            final Predicate<String> domainMaxLength = dm -> dm.getBytes(StandardCharsets.UTF_8).length <= 255;

            return hasMultipleParts.and(domainMaxLength).test(domain);
        };
        final Predicate<String> noWhitespaces = email -> Pattern.matches("[^\\p{Space}]+@[^\\p{Space}]+", email);
        final Predicate<String> punycode = domain -> {
            String[] domainParts;
            try {
                domainParts = IDN.toASCII(domain).split("\\.");
            } catch (IllegalArgumentException e) {
                return false;
            }
            return Arrays.stream(domainParts).allMatch(p -> p.matches("^[a-zA-Z0-9]([a-zA-Z0-9-]*[a-zA-Z0-9]$)?"));
        };

        return nonEmpty.test(trimmedEmailString)
                && noWhitespaces.test(emailString)
                && usernameMaxLength.test(parts[0])
                && validDomain.and(punycode).test(parts[1]);
    }
}
