package com.atlassian.cache.servlet;

import com.atlassian.cache.servlet.handler.CacheExpirationHandler;
import com.atlassian.cache.servlet.handler.LastModifiedCacheExpirationHandler;
import com.atlassian.cache.servlet.resolver.ContentResolver;
import org.apache.commons.collections.MultiHashMap;
import org.apache.commons.collections.MultiMap;
import org.apache.log4j.Logger;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;
import java.util.TreeSet;

public class CombinedCachingServlet extends HttpServlet
{
    protected final Logger log = Logger.getLogger(this.getClass());

    private static final String DEFAULT_CONTENT_TYPE = "text/javascript";
    private static final String CONTENT_RESOLVER_PREFIX = "content.resolver";

    public static final String DEFAULT_ENCODING = "UTF-8";

    private String data = null;
    private CacheExpirationHandler cacheExpirationHandler;
    private String contentType;
    private HashMap contentResolvers;
    private MultiMap contentByResolver;

    public synchronized void init(ServletConfig servletConfig) throws ServletException
    {
        super.init(servletConfig);

        initializeCacheExpirationHandler(servletConfig.getInitParameter("cache.expiration.handler.class"));
        initializeContentType(servletConfig.getInitParameter("content.type"));
        initializeContentResolvers(servletConfig);
        initializeContentParams();
    }

    protected void service(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException
    {
        // always set the content type as defined in the init params
        response.setContentType(contentType);

        // check to see if we need to do any work
        if (getCacheExpirationHandler().checkRequest(request, response))
        {
            return;
        }

        response.getWriter().write(getDataToSend(request, response));
        response.getWriter().flush();
    }

    private String getDataToSend(HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException
    {
        if (isCacheDisabled())
            return makeData(request, response);
        else
            return getCachedData(request, response);
    }

    private synchronized String getCachedData(HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException
    {
        if (data == null)
            data = makeData(request, response);

        return data;
    }

    private boolean isCacheDisabled()
    {
        return "true".equals(System.getProperty("atlassian.disable.caches", "false"));
    }

    private String makeData(HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException
    {
        try
        {
            StringBuffer buffer = new StringBuffer();

            // run through each contentResolver that is registered
            for (Iterator iterator = contentResolvers.keySet().iterator(); iterator.hasNext();)
            {
                String resolverKey = (String) iterator.next();
                // If we have content registered for this resolver
                if (contentByResolver.containsKey(resolverKey))
                {
                    // Run through all the registered content, asking the resolver to resolve it.
                    for (Iterator iterator1 = ((Collection) contentByResolver.get(resolverKey)).iterator(); iterator1.hasNext();)
                    {
                        String path = (String) iterator1.next();
                        ContentResolver resolver = (ContentResolver) contentResolvers.get(resolverKey);
                        buffer.append(resolver.getContent(path, request, response));
                    }
                }
            }
            return buffer.toString();
        }
        catch (IOException e)
        {
            log.error("Error building combined cached servlet data: " + request.getRequestURI() + " - " + e.toString(), e);
            throw e;
        }
        catch (ServletException e)
        {
            log.error("Error building combined cached servlet data: " + request.getRequestURI() + " - " + e.toString(), e);
            throw e;
        }
    }

    private CacheExpirationHandler getCacheExpirationHandler()
    {
        return cacheExpirationHandler;
    }

    /**
     * Run through all the registered content resolver keys and then run through the init params looking for
     * content entries for each resolver key. Store the content path with the resolver key.
     */
    private void initializeContentParams()
    {
        contentByResolver = new MultiHashMap();

        Enumeration e = getInitParameterNames();
        Set sortedNames = new TreeSet();
        while (e.hasMoreElements())
        {
            sortedNames.add(e.nextElement());
        }
        // Run through all the contentResolvers that were registered
        for (Iterator iterator = contentResolvers.keySet().iterator(); iterator.hasNext();)
        {
            String resolverKey = (String) iterator.next();
            // Run through all the init params looking for the resolver key as a prefix
            for (Iterator i = sortedNames.iterator(); i.hasNext();)
            {
                String name = (String) i.next();
                if (name.startsWith(resolverKey))
                {
                    // store the path in the list associated with the resolver key
                    contentByResolver.put(resolverKey, getInitParameter(name));
                }
            }

        }
    }

    /**
     * Runs through all the init params looking for param names starting with CONTENT_RESOLVER_PREFIX. We will use
     * the remainder of the name as the key (which is referenced in the content entries) which will be associated
     * with the implementation of the content resolver (the class specified in the value of the param). If the class
     * is not found then the resolver will not be added.
     * <p/>
     * ex.
     * <pre>
     *  &lt;init-param&gt;
     *    &lt;param-name&gt;content.resolver.dwr&lt;/param-name&gt;
     *    &lt;param-value&gt;com.atlassian.cache.servlet.resolver.DwrContentResolver&lt;/param-value&gt;
     *  &lt;/init-param&gt;
     * </pre>
     * <p/>
     * This will map the key: dwr to the implementation: com.atlassian.cache.servlet.resolver.DwrContentResolver
     */
    private void initializeContentResolvers(ServletConfig config) throws ServletException
    {
        contentResolvers = new HashMap();

        for (Enumeration e = getInitParameterNames(); e.hasMoreElements();)
        {
            String name = (String) e.nextElement();
            if (name.startsWith(CONTENT_RESOLVER_PREFIX))
            {
                Object objResolver = getObjectForClass(getInitParameter(name));
                if (objResolver != null && objResolver instanceof ContentResolver)
                {
                    ContentResolver resolver = (ContentResolver) objResolver;
                    // always make sure to initialize the resolver
                    resolver.init(config);

                    // Get the key to map this resolver from the name and add the impl to the resolvers map
                    String key = name.substring(CONTENT_RESOLVER_PREFIX.length() + 1);
                    contentResolvers.put(key, resolver);
                }
                else
                {
                    log.warn("The class " + getInitParameter(name) + " does not implement ContentResolver.");
                }
            }
        }
    }

    private void initializeContentType(String initParameter)
    {
        if (initParameter == null)
        {
            contentType = DEFAULT_CONTENT_TYPE;
        }
        else
        {
            contentType = initParameter;
        }
    }

    private void initializeCacheExpirationHandler(String cacheExpirationHandlerClass)
    {
        Object objCacheExpirationHandler = getObjectForClass(cacheExpirationHandlerClass);

        if (objCacheExpirationHandler != null && objCacheExpirationHandler instanceof CacheExpirationHandler)
        {
            cacheExpirationHandler = (CacheExpirationHandler) objCacheExpirationHandler;
        }
        else
        {
            log.info("Unable to resolve class: " + cacheExpirationHandlerClass + " as an implementation of CacheExpirationHandler");
            // fallback to the default implementation
            cacheExpirationHandler = new LastModifiedCacheExpirationHandler();
            log.info("Using the LastModifiedCacheExpirationHandler cache expiration handler");
        }
    }

    private Object getObjectForClass(String clazz)
    {
        Object obj = null;
        try
        {
            if (clazz != null)
            {
                obj = Class.forName(clazz).newInstance();
            }
        }
        catch (Exception e)
        {
            log.warn("Unable to resolve class for className: " + clazz);
        }
        return obj;
    }
}
