package com.atlassian.cache.memory;

import com.atlassian.cache.CacheSettingsDefaultsProvider;
import com.atlassian.cache.ManagedCache;
import com.atlassian.cache.impl.jmx.MBeanRegistrar;
import com.atlassian.cache.memory.jmx.MemoryCacheMXBeanRegistrar;
import io.atlassian.util.concurrent.ManagedLock;
import io.atlassian.util.concurrent.ManagedLocks;

import java.util.function.Supplier;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.management.MBeanServer;
import java.util.concurrent.locks.ReentrantReadWriteLock;

/**
 * Extension of @{@link MemoryCacheManager} wrapping it with registering JMX metrics for caches.
 *
 * @since v3.1
 */
public class JMXMemoryCacheManager extends MemoryCacheManager implements MBeanRegistrar
{
    private final MemoryCacheMXBeanRegistrar mbeansRegistrar = new MemoryCacheMXBeanRegistrar();

    private ManagedLock.ReadWrite metricsLock = ManagedLocks.manageReadWrite(new ReentrantReadWriteLock());

    public JMXMemoryCacheManager()
    {
        super();
    }

    public JMXMemoryCacheManager(CacheSettingsDefaultsProvider cacheSettingsDefaultsProvider)
    {
        super(cacheSettingsDefaultsProvider);
    }

    @Override
    protected void putCacheInMap(@Nonnull String name, @Nonnull Supplier<ManagedCache> supplier)
    {
        metricsLock.read().withLock(() ->
        {
            JMXMemoryCacheManager.super.putCacheInMap(name, supplier);
            mbeansRegistrar.registerMBean(name);
        });
    }

    @Override
    public void registerMBeans(@Nullable MBeanServer mBeanServer)
    {
        metricsLock.write().withLock(() ->
        {
            mbeansRegistrar.enableCollectingJMXMetrics(mBeanServer, JMXMemoryCacheManager.this);
        });
    }

    @Override
    public void unregisterMBeans(@Nullable MBeanServer mBeanServer)
    {
        metricsLock.write().withLock(() ->
        {
            mbeansRegistrar.unregisterMBeans();
        });
    }
}
