package com.atlassian.cache.memory;

import com.atlassian.cache.CacheLoader;
import com.atlassian.cache.CacheSettings;
import com.atlassian.cache.CacheSettingsBuilder;
import com.atlassian.cache.CacheSettingsDefaultsProvider;
import com.atlassian.cache.CachedReference;
import com.atlassian.cache.ManagedCache;
import com.atlassian.cache.impl.AbstractCacheManager;
import com.atlassian.cache.impl.ReferenceKey;
import com.atlassian.cache.impl.StrongSupplier;
import com.atlassian.cache.impl.WeakSupplier;
import com.atlassian.instrumentation.DefaultInstrumentRegistry;
import com.atlassian.instrumentation.SimpleTimer;
import com.atlassian.instrumentation.caches.CacheCollector;
import com.atlassian.instrumentation.caches.CacheKeys;
import com.atlassian.util.concurrent.Supplier;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.LoadingCache;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.concurrent.TimeUnit.MILLISECONDS;

/**
 * Maintains a mapping of name -&gt; Cache and provides factory methods for creating and getting caches.
 *
 * @since 2.0
 */
public class MemoryCacheManager extends AbstractCacheManager
{
    @Nonnull
    public MemoryCacheManager()
    {
        super(null);
    }

    public MemoryCacheManager(CacheSettingsDefaultsProvider cacheSettingsDefaultsProvider)
    {
        super(cacheSettingsDefaultsProvider);
    }

    @Nonnull
    @SuppressWarnings("unchecked")
    @Override
    public <K, V> com.atlassian.cache.Cache<K, V> getCache(@Nonnull String name, @Nullable CacheLoader<K, V> loader, @Nonnull CacheSettings settings)
    {
        return getCache(name, loader, settings, DelegatingCache.DEFAULT_CREATE_FUNCTION);
    }

    @Nonnull
    @SuppressWarnings("unchecked")
    public <K, V> com.atlassian.cache.Cache<K, V> getCache(@Nonnull String name, @Nullable CacheLoader<K, V> loader,
                                                           @Nonnull CacheSettings settings,
                                                           @Nonnull final DelegatingCache.CreateFunction createFunction)
    {
        if (null == loader)
        {
            return (com.atlassian.cache.Cache<K, V>) createSimpleCache(name, mergeSettings(name, settings), createFunction);
        }
        else
        {
            return (com.atlassian.cache.Cache<K, V>) createComputingCache(name, mergeSettings(name, settings), loader, createFunction);
        }
    }

    @Nonnull
    @Override
    public <V> CachedReference<V> getCachedReference(@Nonnull final String name,
                                                     @Nonnull final com.atlassian.cache.Supplier<V> supplier,
                                                     @Nonnull final CacheSettings settings)
    {
        return getCachedReference(name, supplier, settings, DelegatingCachedReference.DEFAULT_CREATE_FUNCTION);
    }

    public <V> CachedReference<V> getCachedReference(@Nonnull final String name,
                                                     @Nonnull final com.atlassian.cache.Supplier<V> supplier,
                                                     @Nonnull final CacheSettings settings,
                                                     @Nonnull final DelegatingCachedReference.CreateFunction createFunction)
    {
        // Force the cache settings to be flushable.
        final CacheSettings overridenSettings = mergeSettings(name,
                settings.override(new CacheSettingsBuilder().flushable().build()));

        return cacheCreationLocks.get(name).withLock(new com.atlassian.util.concurrent.Supplier<DelegatingCachedReference<V>>()
        {
            @Override
            public DelegatingCachedReference<V> get()
            {
                // We need to always create a new instance as any old loader may belong to a plugin that has gone away,
                // resulting in a whole world of ClassLoader pain.
                final DelegatingCachedReference.DelegatingReferenceRemovalListener<V> listener = new DelegatingCachedReference.DelegatingReferenceRemovalListener<V>();

                final CacheCollector collector = new DefaultInstrumentRegistry().pullCacheCollector(name);

                LoadingCache<ReferenceKey, V> computingCache = createCacheBuilder(overridenSettings)
                        .removalListener(listener)
                        .build(new com.google.common.cache.CacheLoader<ReferenceKey, V>()
                        {
                            // Wrap the loader with timing code so if stats are enabled we can get numbers.
                            @Override
                            public V load(@Nonnull final ReferenceKey key) throws Exception
                            {
                                if(collector.isEnabled())
                                {
                                    SimpleTimer timer = new SimpleTimer(CacheKeys.LOAD_TIME.toString());
                                    timer.start();
                                    try
                                    {
                                        V value = supplier.get();
                                        listener.onSupply(value);
                                        return value;
                                    }
                                    finally
                                    {
                                        timer.end();
                                        collector.miss();
                                        collector.getSplits().add(timer);
                                    }
                                }
                                else
                                {
                                    V value = supplier.get();
                                    listener.onSupply(value);
                                    return value;
                                }
                            }
                        });

                DelegatingCachedReference<V> cache = createFunction.create(computingCache, name, overridenSettings, collector);
                listener.setCachedReference(cache);
                putCacheInMap(name, new WeakSupplier<ManagedCache>(cache));
                return cache;
            }
        });
    }

    @Override
    protected ManagedCache createSimpleCache(@Nonnull String name, @Nonnull CacheSettings settings) {
        return createSimpleCache(name, settings, DelegatingCache.DEFAULT_CREATE_FUNCTION);
    }

    protected <K,V> ManagedCache createSimpleCache(@Nonnull final String name, @Nonnull final CacheSettings settings,
                                                   @Nonnull final DelegatingCache.CreateFunction createFunction)
    {

        Supplier<ManagedCache> cacheSupplier = caches.get(name);
        if (cacheSupplier != null)
        {
            ManagedCache cache = cacheSupplier.get();
            if (cache != null)
            {
                return cache;
            }
        }
        return cacheCreationLocks.get(name).withLock(new com.atlassian.util.concurrent.Supplier<ManagedCache>()
        {
            @Override
            public ManagedCache get()
            {
                if (!caches.containsKey(name))
                {
                    DelegatingCache.DelegatingRemovalListener<K, V> listener = new DelegatingCache.DelegatingRemovalListener<>();

                    final Cache<K, V> simpleCache = createCacheBuilder(settings)
                            .removalListener(listener)
                            .build();

                    DelegatingCache<K, V> cache = createFunction.create(simpleCache, name, settings, null);
                    listener.setCache(cache);

                    putCacheInMap(name, new StrongSupplier<ManagedCache>(cache));
                }
                return caches.get(name).get();
            }
        });
    }


    @Override
    protected <K, V> ManagedCache createComputingCache(@Nonnull String name, @Nonnull CacheSettings settings, @Nullable CacheLoader<K, V> loader) {
        return createComputingCache(name, settings, loader, DelegatingCache.DEFAULT_CREATE_FUNCTION);
    }

    protected <K, V> ManagedCache createComputingCache(@Nonnull final String name,
                                                       @Nonnull final CacheSettings settings,
                                                       final CacheLoader<K, V> loader,
                                                       @Nonnull final DelegatingCache.CreateFunction createFunction)
    {
        return cacheCreationLocks.get(name).withLock(new com.atlassian.util.concurrent.Supplier<ManagedCache>()
        {
            @Override
            public ManagedCache get()
            {
                // We need to always create a new instance as any old loader may belong to a plugin that has gone away,
                // resulting in a whole world of ClassLoader pain.
                final DelegatingCache.DelegatingRemovalListener<K, V> listener = new DelegatingCache.DelegatingRemovalListener<K, V>();

                final CacheLoader<K, V> wrappedLoader = new CacheLoader<K, V>()
                {
                    @Nonnull
                    @Override
                    public V load(@Nonnull K key)
                    {
                        V value = loader.load(key);
                        listener.onSupply(key, value);
                        return value;
                    }
                };

                final Cache<K, V> simpleCache = createCacheBuilder(settings)
                        .removalListener(listener)
                        .build();

                DelegatingCache<K, V> cache = createFunction.create(simpleCache, name, settings, wrappedLoader);
                listener.setCache(cache);
                putCacheInMap(name, new WeakSupplier<ManagedCache>(cache));
                return cache;
            }
        });
    }

    /**
     * Use this method to store values in caches map to allow read/write synchronization in @{@link JMXMemoryCacheManager}
     *
     * @param name cache name
     * @param supplier supplier containing cache to add
     */
    protected void putCacheInMap(@Nonnull String name, @Nonnull Supplier<ManagedCache> supplier)
    {
        caches.put(name, supplier);
    }

    private static CacheBuilder<Object, Object> createCacheBuilder(CacheSettings settings)
    {
        final CacheBuilder<Object, Object> cacheBuilder = CacheBuilder.newBuilder();
        if (null != settings.getMaxEntries())
        {
            cacheBuilder.maximumSize(settings.getMaxEntries());
        }

        // Old form of stats collection - leave in place. It allows stats from the delegate.
        if (null != settings.getStatisticsEnabled() && settings.getStatisticsEnabled())
        {
            cacheBuilder.recordStats();
        }

        if (null != settings.getExpireAfterAccess())
        {
            cacheBuilder.expireAfterAccess(settings.getExpireAfterAccess(), MILLISECONDS);
        }
        else if (null != settings.getExpireAfterWrite())
        {
            cacheBuilder.expireAfterWrite(settings.getExpireAfterWrite(), MILLISECONDS);
        }
        return cacheBuilder;
    }
}