package com.atlassian.cache.memory;

import com.atlassian.cache.CacheStatisticsKey;
import com.atlassian.instrumentation.caches.CacheCollector;
import com.atlassian.util.concurrent.Supplier;
import com.google.common.cache.Cache;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableSortedMap;

import java.util.SortedMap;

import static com.atlassian.cache.CacheStatisticsKey.EVICTION_COUNT;
import static com.atlassian.cache.CacheStatisticsKey.HIT_COUNT;
import static com.atlassian.cache.CacheStatisticsKey.LOAD_COUNT;
import static com.atlassian.cache.CacheStatisticsKey.LOAD_EXCEPTION_COUNT;
import static com.atlassian.cache.CacheStatisticsKey.LOAD_SUCCESS_COUNT;
import static com.atlassian.cache.CacheStatisticsKey.MISS_COUNT;
import static com.atlassian.cache.CacheStatisticsKey.PUT_COUNT;
import static com.atlassian.cache.CacheStatisticsKey.REMOVE_COUNT;
import static com.atlassian.cache.CacheStatisticsKey.REQUEST_COUNT;
import static com.atlassian.cache.CacheStatisticsKey.SIZE;
import static com.atlassian.cache.CacheStatisticsKey.TOTAL_LOAD_TIME;
import static com.atlassian.cache.CacheStatisticsKey.TOTAL_MISS_TIME;

/**
 * @since v2.4.0
 */
public class DelegatingCacheStatistics
{
    public static SortedMap<CacheStatisticsKey, Supplier<Long>> toStatistics(Cache<?, ?> internalCache)
    {
        ImmutableSortedMap.Builder<CacheStatisticsKey, Supplier<Long>> map = ImmutableSortedMap.<CacheStatisticsKey, Supplier<Long>>orderedBy(CacheStatisticsKey.SORT_BY_LABEL)
                .put(SIZE, memoize(internalCache.size()))
                .put(HIT_COUNT, memoize(internalCache.stats().hitCount()))
                .put(MISS_COUNT, memoize(internalCache.stats().missCount()))
                .put(LOAD_COUNT, memoize(internalCache.stats().loadCount()))
                .put(LOAD_SUCCESS_COUNT, memoize(internalCache.stats().loadSuccessCount()))
                .put(LOAD_EXCEPTION_COUNT, memoize(internalCache.stats().loadExceptionCount()))
                .put(EVICTION_COUNT, memoize(internalCache.stats().evictionCount()))
                .put(TOTAL_LOAD_TIME, memoize(internalCache.stats().totalLoadTime()))
                .put(REQUEST_COUNT, memoize(internalCache.stats().requestCount()));

        // don't bother reporting this statistic as "zero" for caches that
        // can't even calculate it. Only LoadingCaches calculate it.
        if (internalCache instanceof LoadingCache)
        {
            map.put(TOTAL_MISS_TIME, memoize(internalCache.stats().totalLoadTime()));
        }

        return map.build();
    }

    public static SortedMap<CacheStatisticsKey, Supplier<Long>> toStatistics(CacheCollector counter)
    {
        ImmutableSortedMap.Builder<CacheStatisticsKey, Supplier<Long>> map =
                ImmutableSortedMap.<CacheStatisticsKey, Supplier<Long>>orderedBy(CacheStatisticsKey.SORT_BY_LABEL)
                        .put(HIT_COUNT, memoize(counter.getHits()))
                        .put(MISS_COUNT, memoize(counter.getMisses()))
                        .put(TOTAL_MISS_TIME, memoize(counter.getMissTime()))
                        .put(PUT_COUNT, memoize(counter.getPuts()))
                        .put(SIZE, memoize(counter.getCacheSize()))
                        .put(REMOVE_COUNT, memoize(counter.getRemoves()));

        return map.build();
    }

    private static <V> Supplier<V> memoize(final V value)
    {
        return new ImmediateSupplier<V>(value);
    }

    static class ImmediateSupplier<V> implements Supplier<V>
    {
        private final V value;

        ImmediateSupplier(final V value)
        {
            this.value = value;
        }

        @Override
        public V get()
        {
            return value;
        }

        @Override
        public String toString()
        {
            return "ImmediateSupplier[" + value + ']';
        }
    }
}
