package com.atlassian.cache.compat.guava11.memory;

import com.atlassian.cache.compat.Cache;
import com.atlassian.cache.compat.CacheException;
import com.atlassian.cache.compat.CacheSettings;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ComputationException;

import java.util.Collection;
import javax.annotation.Nullable;

/**
 * A Cache that delegates LoadingCache.
 *
 * @since v1.0
 */
class MemoryCache<K,V> implements Cache<K,V>
{
    private final String name;
    private final com.google.common.cache.Cache<K, V> internalCache;

    private MemoryCache(final com.google.common.cache.Cache<K, V> internalCache, String name)
    {
        this.internalCache = internalCache;
        this.name = name;
    }

    static <K, V> MemoryCache<K, V> create(final com.google.common.cache.Cache<K, V> internalCache, String name, CacheSettings settings)
    {
        return new MemoryCache<K, V>(internalCache, name);
    }


    @Override
    public String getName()
    {
        return name;
    }

    @Override
    public Collection<K> getKeys()
    {
        try
        {
            return internalCache.asMap().keySet();
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public void put(final K key, final V value)
    {
        try
        {
            internalCache.put(key, value);
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public V get(final K key)
    {
        try
        {
            // need to do some mucking around here - Guava has removed get
            if (internalCache instanceof LoadingCache)
            {
                return ((LoadingCache<K,V>)internalCache).get(key);
            }
            return internalCache.asMap().get(key);
        }
        catch (ComputationException e)
        {
            throw new CacheException(e.getCause());
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public void remove(final K key)
    {
        try
        {
            internalCache.invalidate(key);
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public void removeAll()
    {
        try
        {
            internalCache.invalidateAll();
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public boolean equals(@Nullable final Object other)
    {
        if (other instanceof MemoryCache)
        {
            MemoryCache otherMemoryCache = (MemoryCache) other;
            if (internalCache.equals(otherMemoryCache.internalCache))
            {
                return true;
            }
        }
        return false;
    }

    @Override
    public int hashCode()
    {
        return 3 + internalCache.hashCode();
    }
}
