package com.atlassian.cache.ehcache.replication.rmi;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.atlassian.cache.CacheSettings;
import com.atlassian.cache.ehcache.replication.EhCacheReplicatorConfigFactory;

import net.sf.ehcache.config.CacheConfiguration;

/**
 * An implementation of {@link EhCacheReplicatorConfigFactory} that configures RMI replication.
 *
 * @since 2.6.0
 */
@ParametersAreNonnullByDefault
public class RMICacheReplicatorConfigFactory implements EhCacheReplicatorConfigFactory
{
    // These are all the flags supported by RMICacheReplicatorFactory
    private static final String CACHE_PROPERTIES =
            "replicateAsynchronously=%s," +
                    "replicatePuts=%s," +
                    "replicatePutsViaCopy=%s," +
                    "replicateUpdates=%s," +
                    "replicateUpdatesViaCopy=%s," +
                    "replicateRemovals=true";

    @Override
    @Nonnull
    public CacheConfiguration.CacheEventListenerFactoryConfiguration createCacheReplicatorConfiguration(final CacheSettings settings, final boolean selfLoadingCache)
    {
        final boolean replicateAsynchronously = settings.getReplicateAsynchronously(false);
        final boolean replicateViaCopy = settings.getReplicateViaCopy(false);

        // If the cache is self loading there is no need to replicate puts or updates as the loader takes care of
        // populating the cache as needed.
        // Else if the cache is replicating via copy then puts and updates should be replicated.
        // Else replicating puts is dangerous as it may lead to invalidations ping ponging between nodes. Whether to
        // replicate updates is ambiguous, but setting it may help to prevent stale caches while being less vulnerable
        // to the invalidation storm that comes with simply replicating puts.
        final boolean replicatePuts = !selfLoadingCache && replicateViaCopy;
        final boolean replicateUpdates = !selfLoadingCache;

        final String cacheProperties =
                String.format(CACHE_PROPERTIES, replicateAsynchronously, replicatePuts, replicateViaCopy, replicateUpdates, replicateViaCopy);
        return new CacheConfiguration.CacheEventListenerFactoryConfiguration()
                .className(RMICacheReplicatorFactory.class.getName())
                .properties(cacheProperties);
    }
}
