package com.atlassian.cache.ehcache;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.atlassian.cache.CacheSettings;
import com.atlassian.cache.ehcache.replication.EhCacheReplicatorConfigFactory;
import com.atlassian.cache.ehcache.replication.rmi.RMICacheReplicatorConfigFactory;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Ehcache;
import net.sf.ehcache.config.CacheConfiguration;
import net.sf.ehcache.config.CacheConfiguration.CacheEventListenerFactoryConfiguration;
import net.sf.ehcache.config.PersistenceConfiguration;

import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static java.util.concurrent.TimeUnit.SECONDS;
import static net.sf.ehcache.config.PersistenceConfiguration.Strategy.NONE;

/**
 * Helper for building EhCache caches
 *
 * @since 2.0
 */
@ParametersAreNonnullByDefault
class EhCacheHelper
{
    private static final Logger log = LoggerFactory.getLogger(EhCacheHelper.class);

    static final PersistenceConfiguration.Strategy PERSISTENCE_STRATEGY = NONE;

    private static final PersistenceConfiguration PERSISTENCE_CONFIGURATION =
            new PersistenceConfiguration().strategy(PERSISTENCE_STRATEGY);

    private final @Nullable
    EhCacheReplicatorConfigFactory replicatorConfigFactory;

    /**
     * @deprecated Since 2.6.0 Use {@link #EhCacheHelper(EhCacheReplicatorConfigFactory)}
     */
    @Deprecated
    EhCacheHelper()
    {
        this(new RMICacheReplicatorConfigFactory());
    }

    EhCacheHelper(@Nullable EhCacheReplicatorConfigFactory replicatorConfigFactory)
    {
        this.replicatorConfigFactory = replicatorConfigFactory;
    }

    private CacheEventListenerFactoryConfiguration getCacheEventListenerFactoryConfiguration(
            final CacheSettings settings, final boolean selfLoading)
    {
        if (replicatorConfigFactory != null)
        {
            return replicatorConfigFactory.createCacheReplicatorConfiguration(settings, selfLoading);
        }
        else
        {
            throw new IllegalStateException("No EhCacheReplicatorConfigFactory has been configured");
        }
    }

    @Nonnull
    Ehcache getEhcache(final String name, final CacheManager ehMgr, final CacheSettings settings, final boolean selfLoading, final boolean statisticsEnabled)
    {
        //Create a Cache specifying its configuration.
        CacheConfiguration config = ehMgr.getConfiguration().getDefaultCacheConfiguration().clone()
                .name(name)
                .statistics(statisticsEnabled)
                .persistence(PERSISTENCE_CONFIGURATION);

        final boolean replicateCache = isReplicateCache(ehMgr, settings);
        if (replicateCache)
        {
            config.cacheEventListenerFactory(
                    getCacheEventListenerFactoryConfiguration(settings, selfLoading));
        }

        if (null != settings.getMaxEntries())
        {
            config.setMaxEntriesLocalHeap(settings.getMaxEntries());
        }

        // Cache should not be eternal if expiry has been set
        if (settings.getExpireAfterAccess() != null || settings.getExpireAfterWrite() != null)
        {
            config.setEternal(false);
        }

        if (null != settings.getExpireAfterAccess())
        {
            config.timeToIdleSeconds(SECONDS.convert(settings.getExpireAfterAccess(), MILLISECONDS));
        }

        if (null != settings.getExpireAfterWrite())
        {
            config.timeToLiveSeconds(SECONDS.convert(settings.getExpireAfterWrite(), MILLISECONDS));
        }

        Ehcache cache = new net.sf.ehcache.Cache(config);
        if (selfLoading)
        {
            cache = new SynchronizedLoadingCacheDecorator(cache);
        }
        return ehMgr.addCacheIfAbsent(cache);
    }

    private boolean isReplicateCache(final CacheManager ehMgr, final CacheSettings settings)
    {
        final boolean isLocalCacheSetting = settings.getLocal(false); // default to non-local as per API

        if (!isLocalCacheSetting)
        {
            if (replicatorConfigFactory != null)
            {
                final boolean hasPeerProvider = !ehMgr.getCacheManagerPeerProviders().isEmpty();
                if (hasPeerProvider)
                {
                    return true;
                }
                else
                {
                    log.warn("No PeerProviders configured in ehcache, replication cannot be configured for non-local cache");
                }
            }
        }
        return false;
    }
}
