package com.atlassian.cache.ehcache;

import java.util.SortedMap;

import com.atlassian.cache.CacheStatisticsKey;
import com.atlassian.util.concurrent.Supplier;

import com.google.common.collect.ImmutableSortedMap;

import net.sf.ehcache.statistics.StatisticsGateway;

import static com.atlassian.cache.CacheStatisticsKey.EVICTION_COUNT;
import static com.atlassian.cache.CacheStatisticsKey.HEAP_SIZE;
import static com.atlassian.cache.CacheStatisticsKey.HIT_COUNT;
import static com.atlassian.cache.CacheStatisticsKey.MISS_COUNT;
import static com.atlassian.cache.CacheStatisticsKey.PUT_COUNT;
import static com.atlassian.cache.CacheStatisticsKey.REMOVE_COUNT;
import static com.atlassian.cache.CacheStatisticsKey.SIZE;
import static com.atlassian.util.concurrent.Suppliers.memoize;

/**
 * @since v2.4.0
 */
public class DelegatingCacheStatistics
{
    public static SortedMap<CacheStatisticsKey, Supplier<Long>> toStatistics(final StatisticsGateway stats)
    {
        // Note: While most of these statistics we can reasonably expect to be cheap and just memoize, the
        // heap size in bytes is very likely to be expensive.  To make it possible to opt out of retrieving
        // it, we hide it behind a Supplier, here.
        final Supplier<Long> heapSize = new Supplier<Long>()
        {
            @Override
            public Long get()
            {
                return stats.getLocalHeapSizeInBytes();
            }
        };

        return ImmutableSortedMap.<CacheStatisticsKey,Supplier<Long>>orderedBy(CacheStatisticsKey.SORT_BY_LABEL)
                .put(SIZE, memoize(stats.getSize()))
                .put(HEAP_SIZE, heapSize)
                .put(HIT_COUNT, memoize(stats.cacheHitCount()))
                .put(PUT_COUNT, memoize(stats.cachePutCount()))
                .put(REMOVE_COUNT, memoize(stats.cacheRemoveCount()))
                .put(MISS_COUNT, memoize(stats.cacheMissCount()))
                .put(EVICTION_COUNT, memoize(stats.cacheEvictedCount()))
                .build();
    }
}
