package com.atlassian.cache.compat.delegate;

import com.atlassian.cache.CacheFactory;
import com.atlassian.cache.compat.Cache;
import com.atlassian.cache.compat.CacheLoader;
import com.atlassian.cache.compat.CacheSettings;
import com.atlassian.cache.compat.CachedReference;
import com.atlassian.cache.compat.Supplier;
import com.atlassian.sal.api.component.ComponentLocator;
import com.atlassian.util.concurrent.NotNull;

import com.google.common.annotations.VisibleForTesting;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.atlassian.cache.compat.delegate.CacheSettingsMapper.mapCacheSettings;
import static com.atlassian.cache.compat.delegate.DelegatingCache.wrapCache;
import static com.atlassian.cache.compat.delegate.DelegatingCacheLoader.wrapCacheLoader;
import static com.atlassian.cache.compat.delegate.DelegatingCachedReference.wrapCachedReference;
import static com.atlassian.cache.compat.delegate.DelegatingSupplier.wrapSupplier;
import static com.atlassian.util.concurrent.Assertions.notNull;

/**
 * @since v1.0
 */
public class DelegatingCacheFactory implements com.atlassian.cache.compat.CacheFactory
{
    private static final Logger LOG = LoggerFactory.getLogger(DelegatingCacheFactory.class);

    // MUST be referenced by name!  Any direct reference to the CacheManager class will mess up transformed plugins.
    private static final String CACHE_MANAGER_CLASS = "com.atlassian.cache.CacheManager";

    private final CacheFactory delegate;

    @SuppressWarnings("UnusedDeclaration")  // Accessed via reflection
    public DelegatingCacheFactory()
    {
        this(getCacheFactory());
    }

    @VisibleForTesting
    DelegatingCacheFactory(com.atlassian.cache.CacheFactory delegate)
    {
        this.delegate = notNull("delegate", delegate);
    }

    @Override
    public <V> CachedReference<V> getCachedReference(@NotNull String name, @NotNull Supplier<V> supplier)
    {
        return wrapCachedReference(delegate.getCachedReference(name, wrapSupplier(supplier)));
    }

    @Override
    public <V> CachedReference<V> getCachedReference(@NotNull String name, @NotNull Supplier<V> supplier,
            @NotNull CacheSettings required)
    {
        return wrapCachedReference(delegate.getCachedReference(name, wrapSupplier(supplier),
                mapCacheSettings(required)));
    }

    @Override
    public <V> CachedReference<V> getCachedReference(@NotNull Class<?> owningClass, @NotNull String name,
            @NotNull Supplier<V> supplier)
    {
        return wrapCachedReference(delegate.getCachedReference(owningClass, name, wrapSupplier(supplier)));
    }

    @Override
    public <V> CachedReference<V> getCachedReference(@NotNull Class<?> owningClass, @NotNull String name,
            @NotNull Supplier<V> supplier, @NotNull CacheSettings required)
    {
        return wrapCachedReference(delegate.getCachedReference(owningClass, name, wrapSupplier(supplier),
                mapCacheSettings(required)));
    }

    @Override
    public <K, V> Cache<K, V> getCache(@NotNull String name)
    {
        return wrapCache(delegate.<K,V>getCache(name));
    }

    @Override
    public <K, V> Cache<K, V> getCache(@NotNull Class<?> owningClass, @NotNull String name)
    {
        return wrapCache(delegate.<K,V>getCache(owningClass, name));
    }

    @Override
    public <K, V> Cache<K, V> getCache(@NotNull String name, CacheLoader<K, V> loader)
    {
        return wrapCache(delegate.getCache(name, wrapCacheLoader(loader)));
    }

    @Override
    public <K, V> Cache<K, V> getCache(@NotNull String name, CacheLoader<K, V> loader, @NotNull CacheSettings required)
    {
        return wrapCache(delegate.getCache(name, wrapCacheLoader(loader), mapCacheSettings(required)));
    }

    @Override
    public <K, V> Cache<K, V> getCache(@NotNull String name, @NotNull Class<K> keyType, @NotNull Class<V> valueType)
    {
        return wrapCache(delegate.getCache(name, keyType, valueType));
    }



    private static CacheFactory getCacheFactory()
    {
        final Class<?> cacheManagerClass;
        try
        {
            cacheManagerClass = DelegatingCacheFactory.class.getClassLoader().loadClass(CACHE_MANAGER_CLASS);
        }
        catch (ClassNotFoundException e)
        {
            throw new IllegalStateException("The CacheLoader class exists, but the CacheManager class does not?!" , e);
        }

        final Object cacheManager = ComponentLocator.getComponent(cacheManagerClass);
        if (cacheManager != null)
        {
            LOG.debug("Delegating to atlassian-cache-api v2.x");
            return (CacheFactory)cacheManager;
        }

        final CacheFactory cacheFactory = ComponentLocator.getComponent(CacheFactory.class);
        if (cacheFactory != null)
        {
            LOG.debug("Delegating to atlassian-cache-api v2.x (CacheFactory)");
            return cacheFactory;
        }

        throw new IllegalStateException("The CacheLoader class exists, but neither CacheManager nor CacheFactory is " +
                "registered in the container.  Maybe you bundled atlassian-cache-api v2.x?  Don't do that.");
    }
}

