package com.atlassian.cache.compat;

import com.atlassian.annotations.PublicApi;

import javax.annotation.Nonnull;
import java.util.Collection;

@PublicApi
public interface Cache<K, V>
{
    /**
     * The name of the cache, uniquely identifies this cache.
     *
     * @return the name of the cache.
     */
    String getName();

    /**
     * Gets the keys of all objects currently stored in the cache. This will return the keys in a new collection.
     * @return a collection of {@link Object}s keys.
     */
    Collection<K> getKeys();

    /**
     * Retrieve an object from this cache.
     *
     * A Cache created using a {@link CacheLoader} will never return null.
     *
     * @param key uniquely identifying the object to be retrieved.
     * @return the object from the cache, or <code>null</code> if the object is not found.
     */
    V get(@Nonnull final K key);

    /**
     * Put an object into the cache. If the specified key already exists within the cache, it will be replaced by the
     * new object.
     * <p>
     * <b>NOTE:</b> Users of caches that wish to be well behaved in a clustered environment should use the
     * {@link CacheLoader} semantics and supply a {@link CacheLoader} when getting the {@link Cache}.
     *
     * @param key uniquely identifying the object to be added into the cache.
     * @param value to be cached.
     */
    void put(@Nonnull final K key, final V value);

    /**
     * Remove the object identified by the key from the cache. If no object can be found associated with this key then
     * no action is taken.
     *
     * @param key uniquely identifying the object to be removed.
     */
    void remove(@Nonnull final K key);

    /**
     * Remove all of the objects from this cache.
     */
    void removeAll();
}
