package com.atlassian.cache.compat;

import java.util.concurrent.TimeUnit;
import com.atlassian.annotations.PublicApi;
import com.atlassian.util.concurrent.NotNull;

/**
 * A builder for creating {@link CacheSettings} instances.
 *
 * @since v1.0
 */
@PublicApi
public class CacheSettingsBuilder
{
    private Long expireAfterAccess;
    private Long expireAfterWrite;
    private Boolean flushable;
    private Boolean local;
    private Integer maxEntries;
    private Boolean replicateAsynchronously;
    private Boolean replicateViaCopy;

    /**
     * @return a new {@link CacheSettings} instance with
     */
    public CacheSettings build()
    {
        return new DefaultCacheSettings(expireAfterAccess, expireAfterWrite, flushable,
                                        local, maxEntries, replicateAsynchronously, replicateViaCopy);
    }

    /**
     * Set a hint for the cache regarding how long entries should be held in the cache.
     * @param expireAfter Time to retain entries for since their last access.
     * @param timeUnit The {@link TimeUnit} for the time
     * @return this builder
     */
    public CacheSettingsBuilder expireAfterAccess(long expireAfter, @NotNull TimeUnit timeUnit)
    {
        this.expireAfterAccess = timeUnit.toMillis(expireAfter);
        return this;
    }

    /**
     * Set a hint for the cache regarding how long entries should be held in the cache.
     * @param expireAfter Time to retain entries from when they were created.
     * @param timeUnit The {@link TimeUnit} for the time
     * @return this builder
     */
    public CacheSettingsBuilder expireAfterWrite(long expireAfter, @NotNull TimeUnit timeUnit)
    {
        this.expireAfterWrite = timeUnit.toMillis(expireAfter);
        return this;
    }

    /**
     * Indicates that this cache can be flushed by the cache manager when required or desired.
     * @return this builder
     */
    public CacheSettingsBuilder flushable()
    {
        this.flushable = true;
        return this;
    }

    /**
     * Indicates that this cache cannot be flushed by the cache manager when required or desired.
     * @return this builder
     */
    public CacheSettingsBuilder unflushable()
    {
        this.flushable = false;
        return this;
    }

    /**
     * Indicates that this cache can have a maximum number of entries.
     * @param maxEntries The maximum number of entries. Must be greater than zero.
     * @return this builder
     */
    public CacheSettingsBuilder maxEntries(int maxEntries)
    {
        if (0 >= maxEntries)
        {
            throw new IllegalArgumentException("maxEntries must be greater than zero, passed: " + maxEntries);
        }
        this.maxEntries = maxEntries;
        return this;
    }

    /**
     * Indicates that in a clustered environment with replicated caches, this
     * cache should replicate asynchronously. By default, it is up to the host
     * application as to whether replication is synchronous or asynchronous.
     * <p>
     * When it is synchronous, cache mutations block on the originating node until
     * all nodes in the cluster have replicated the mutation. Asynchronous
     * replication provides greater responsiveness at the expense of
     * consistency.
     *
     * @return this builder
     */
    public CacheSettingsBuilder replicateAsynchronously()
    {
        this.replicateAsynchronously = true;
        return this;
    }

    /**
     * Indicates that in a clustered environment with replicated caches, this
     * cache should replicate synchronously.
     *
     * @return this builder
     * @see #replicateAsynchronously
     * @since 1.1
     */
    public CacheSettingsBuilder replicateSynchronously()
    {
        this.replicateAsynchronously = false;
        return this;
    }

    /**
     * Indicates that in a clustered environment with replicated caches, this
     * cache should replicate put and update operations by copying the relevant
     * key and value across the wire (requiring both of them to be {@link
     * java.io.Serializable}).
     * <p>
     * By default, it is up to the host application as to whether replication is via
     * copy or invalidation.
     *
     * @return this builder
     * @see #replicateViaInvalidation()
     */
    public CacheSettingsBuilder replicateViaCopy()
    {
        this.replicateViaCopy = true;
        return this;
    }

    /**
     * Indicates that in a clustered environment with replicated caches, this
     * cache should replicate by sending only the key across the wire, for invalidation by
     * the other nodes in the cluster; this requires only the key to be {@link
     * java.io.Serializable}.
     * <br/>
     * By default, it is up to the host application as to whether replication is via
     * copy or invalidation.
     *
     * @return this builder
     * @see #replicateViaCopy
     * @since 1.1
     */
    public CacheSettingsBuilder replicateViaInvalidation()
    {
        this.replicateViaCopy = false;
        return this;
    }

    /**
     * Indicates that this cache should be local to the node (JVM) where the cache is created.
     * By default, caches will be clustered in a clustered deployment.
     * @return this builder
     */
    public CacheSettingsBuilder local()
    {
        this.local = true;
        return this;
    }

    /**
     * Indicates that this cache should be clustered in a clustered deployment.
     * @return this builder
     */
    public CacheSettingsBuilder remote()
    {
        this.local = false;
        return this;
    }
}
