package com.atlassian.cache.impl;

import com.atlassian.cache.Cache;
import com.atlassian.cache.CacheLoader;
import com.atlassian.cache.CacheManager;
import com.atlassian.cache.CacheSettings;
import com.atlassian.cache.CacheSettingsBuilder;
import com.atlassian.cache.CacheSettingsDefaultsProvider;
import com.atlassian.cache.CachedReference;
import com.atlassian.cache.ManagedCache;
import com.atlassian.cache.impl.metrics.CacheManagerMetricEmitter;
import io.atlassian.util.concurrent.ManagedLock;
import io.atlassian.util.concurrent.ManagedLocks;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Function;
import java.util.function.Supplier;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Objects.requireNonNull;

/**
 * A partial implementation of {@link CacheManager}.
 *
 * @since 2.0.6
 */
@ParametersAreNonnullByDefault
public abstract class AbstractCacheManager implements CacheManager
{
    private final CacheManagerMetricEmitter cacheManagerMetricEmitter;

    /**
     * Map of all the caches.
     * Before creating a cache to put into the map, acquire a lock using {@link #cacheCreationLocks}
     * to ensure that a cache is created only once.
     */
    protected final ConcurrentMap<String, Supplier<ManagedCache>> caches = new ConcurrentHashMap<>();

    /** Used to synchronize the creation of caches. */
    protected final Function<String, ManagedLock> cacheCreationLocks = ManagedLocks.weakManagedLockFactory();

    protected final @Nullable CacheSettingsDefaultsProvider cacheSettingsDefaultsProvider;

    protected AbstractCacheManager(@Nullable CacheSettingsDefaultsProvider cacheSettingsDefaultsProvider, @Nonnull CacheManagerMetricEmitter cacheManagerMetricEmitter)
    {
        this.cacheSettingsDefaultsProvider = cacheSettingsDefaultsProvider;
        this.cacheManagerMetricEmitter = requireNonNull(cacheManagerMetricEmitter, "cacheManagerMetricEmitter");
    }

    @Nonnull
    @SuppressWarnings("unchecked")
    @Override
    public Collection<Cache<?, ?>> getCaches()
    {
        List<Cache<?, ?>> managedCaches = new ArrayList<>(64);

        for (final Supplier<ManagedCache> cacheRef : caches.values())
        {
            final ManagedCache managedCache = cacheRef.get();
            // Only return instances of Cache. Do not return CachedReference instances
            if (managedCache instanceof Cache)
            {
                managedCaches.add((Cache<?,?>)managedCache);
            }
        }
        return managedCaches;
    }

    @Nonnull
    @Override
    public Collection<ManagedCache> getManagedCaches()
    {
        List<ManagedCache> managedCaches = new ArrayList<>(64);
        for (final Supplier<ManagedCache> cacheRef : caches.values())
        {
            final ManagedCache managedCache = cacheRef.get();
            if (managedCache != null)
            {
                managedCaches.add(managedCache);
            }
        }
        return managedCaches;
    }

    @Nullable
    @Override
    public ManagedCache getManagedCache(@Nonnull String name)
    {
        final Supplier<ManagedCache> cacheRef = caches.get(name);
        return (null == cacheRef) ? null : cacheRef.get();
    }

    @Override
    public void flushCaches()
    {
        cacheManagerMetricEmitter.emitCacheManagerFlushAll(getClass().getName());
        for (final Supplier<ManagedCache> cacheRef : caches.values())
        {
            final ManagedCache managedCache = cacheRef.get();
            if (managedCache != null && managedCache.isFlushable())
            {
                managedCache.clear();
            }
        }
    }

    @Nonnull
    @Override
    public <K, V> Cache<K, V> getCache(@Nonnull final String name)
    {
        return getCache(name, null);
    }

    @Nonnull
    @Override
    public <K, V> Cache<K, V> getCache(@Nonnull Class<?> owningClass, @Nonnull String name)
    {
        return getCache(cacheName(owningClass, name));
    }

    @Nonnull
    @Override
    public <K, V> Cache<K, V> getCache(@Nonnull final String name, @Nonnull final Class<K> keyType, @Nonnull final Class<V> valueType)
    {
        return getCache(name);
    }

    @Nonnull
    @Override
    public <K, V> Cache<K, V> getCache(@Nonnull String name, @Nullable CacheLoader<K, V> loader)
    {
        return getCache(name, loader, new CacheSettingsBuilder().build());
    }

    @Nonnull
    @Override
    public <V> CachedReference<V> getCachedReference(@Nonnull String name, @Nonnull com.atlassian.cache.Supplier<V> supplier)
    {
        return getCachedReference(name, supplier, new CacheSettingsBuilder().build());
    }

    @Nonnull
    @Override
    public <V> CachedReference<V> getCachedReference(@Nonnull Class<?> owningClass,
                                                     @Nonnull String name,
                                                     @Nonnull com.atlassian.cache.Supplier<V> supplier)
    {
        return getCachedReference(owningClass, name, supplier, new CacheSettingsBuilder().build());
    }

    @Nonnull
    @Override
    public <V> CachedReference<V> getCachedReference(@Nonnull Class<?> owningClass,
                                                     @Nonnull String name,
                                                     @Nonnull com.atlassian.cache.Supplier<V> supplier,
                                                     @Nonnull CacheSettings settings)
    {
        return getCachedReference(cacheName(owningClass, name), supplier, settings);
    }

    private static String cacheName(Class<?> owningClass, String name)
    {
        checkNotNull(name, "name cannot be null");
        return owningClass.getName() + '.' + name;
    }

    @Nonnull
    @SuppressWarnings("unchecked")
    @Override
    public <K, V> Cache<K, V> getCache(@Nonnull String name, @Nullable CacheLoader<K, V> loader, @Nonnull CacheSettings settings)
    {
        if (null == loader)
        {
            return (Cache<K, V>) createSimpleCache(name, mergeSettings(name, settings));
        }
        else
        {
            return (Cache<K, V>) createComputingCache(name, mergeSettings(name, settings), loader);
        }
    }

    /**
     * Creates a cache that upon a miss is able to populate itself using the loader.
     *
     * @return a non-null cache
     */
    protected abstract <K, V> ManagedCache createComputingCache(@Nonnull String name, @Nonnull CacheSettings settings, @Nullable CacheLoader<K, V> loader);

    /**
     * Creates a cache with no loader, i.e. one populated via explicit puts.
     *
     * @param name the name to give the cache (required)
     * @return a non-null cache
     */
    protected abstract ManagedCache createSimpleCache(@Nonnull String name, @Nonnull CacheSettings settings);

    protected CacheSettings mergeSettings(String name, CacheSettings settings)
    {
        if (cacheSettingsDefaultsProvider == null)
        {
            return settings;
        }

        return cacheSettingsDefaultsProvider.getDefaults(name).override(settings);
    }

    @Override
    public void shutdown()
    {
        //By default cacheManages does not have to implements the shutdown method.
    }
}
