package com.atlassian.cache;

import javax.annotation.Nonnull;

import com.atlassian.annotations.PublicApi;

import java.util.Optional;

/**
 * A Resettable reference.
 *
 * @since 2.0
 */
@PublicApi
public interface CachedReference<V>
{
    /**
     * Gets a value from the cache.
     *
     * @return the cached value
     */
    @Nonnull
    V get();

    /**
     * Resets (clears/invalidates) this reference.
     */
    void reset();

    /**
     * Checks if there is currently a value in the cache.
     * <p>
     * Note that cache entries could expire at any time, meaning that even after this method returns true, the reference
     * may still have to be reinitialized on the next call to {@link #get()}.
     * </p>
     * @return true iff {@link #get()} would return a reference without invoking the underlying supplier.
     * @since 5.1
     */
    boolean isPresent();

    /**
     * Checks if there is currently a value in the cache, and if so returns it.
     * <p>
     * Note that cache entries could expire at any time, meaning that even if this method returns a value, the reference
     * may still have to be reinitialized on the next call to {@link #get()}.
     * </p>
     * @return some value iff {@link #get()} would return a reference without invoking the underyling supplier,
     * otherwise empty.
     * @since 5.1
     */
    @Nonnull
    Optional<V> getIfPresent();

    /**
     * Adds a {@link CachedReferenceListener}
     * @param listener the listener
     * @param includeValues if the events sent to this listener will include old/new value. This can be used in cases
     * when the cost of finding these values is big (network sync) but the listener is not interested in the concrete
     * values for events its getting. The support for this parameter is optional and implementation dependent
     *
     * @throws UnsupportedOperationException if not supported
     * @since 2.4
     */
    void addListener(@Nonnull CachedReferenceListener<V> listener, boolean includeValues);

    /**
     * Removes a {@link CachedReferenceListener}
     * @param listener the listener
     *
     * @throws UnsupportedOperationException if not supported
     * @since 2.4
     */
    void removeListener(@Nonnull CachedReferenceListener<V> listener);
}
