define('bitbucket/internal/util/store/reducers', ['exports', 'icepick', 'lodash', 'redux', 'bitbucket/internal/util/get-id', 'bitbucket/internal/util/object'], function (exports, _icepick, _lodash, _redux, _getId, _object) {
    'use strict';

    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.singleEntityRequestReducer = exports.keyByPayloadId = exports.actionTypeIn = exports.isRequestType = exports.keyByRequestProps = exports.keyByRequestProp = exports.getKeyFromProp = exports.getRequestPayload = exports.composeReducers = exports.reduceIf = exports.errorsReducer = exports.getRequestReducerByType = exports.getRequestPageReducersByType = exports.getRequestReducer = exports.getRequestPageReducer = exports._legacyPagingReducer = exports.entitiesReducer = exports.toggleReducer = exports.versionBy = exports.keyBy = exports.reduceByTypes = exports.reduceByType = exports.preserveIfEquivalent = exports.preserveEquivalentArraysUnion = exports.preserveEqualArrays = exports.EMPTY_OBJECT = exports.EMPTY_ARRAY = exports.UNKEYED = undefined;

    function _toConsumableArray(arr) {
        if (Array.isArray(arr)) {
            for (var i = 0, arr2 = Array(arr.length); i < arr.length; i++) {
                arr2[i] = arr[i];
            }

            return arr2;
        } else {
            return Array.from(arr);
        }
    }

    function _defineProperty(obj, key, value) {
        if (key in obj) {
            Object.defineProperty(obj, key, {
                value: value,
                enumerable: true,
                configurable: true,
                writable: true
            });
        } else {
            obj[key] = value;
        }

        return obj;
    }

    var _slicedToArray = function () {
        function sliceIterator(arr, i) {
            var _arr = [];
            var _n = true;
            var _d = false;
            var _e = undefined;

            try {
                for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) {
                    _arr.push(_s.value);

                    if (i && _arr.length === i) break;
                }
            } catch (err) {
                _d = true;
                _e = err;
            } finally {
                try {
                    if (!_n && _i["return"]) _i["return"]();
                } finally {
                    if (_d) throw _e;
                }
            }

            return _arr;
        }

        return function (arr, i) {
            if (Array.isArray(arr)) {
                return arr;
            } else if (Symbol.iterator in Object(arr)) {
                return sliceIterator(arr, i);
            } else {
                throw new TypeError("Invalid attempt to destructure non-iterable instance");
            }
        };
    }();

    var tryFreeze = function tryFreeze(o) {
        return o !== Object(o) ? o : (0, _icepick.freeze)(o);
    };
    var UNKEYED = exports.UNKEYED = '@@_UNKEYED_@@';
    var EMPTY_ARRAY = exports.EMPTY_ARRAY = [];
    var EMPTY_OBJECT = exports.EMPTY_OBJECT = {};

    var preserveEqualArrays = exports.preserveEqualArrays = function preserveEqualArrays(targetVal, sourceVal) {
        // By default, icepick.merge always overwrites arrays, even if they are 'equal'.
        // We incur a small performance hit here to do the isEqual check on Arrays,
        // but it allows us to preserve object references and prevent re-rendering later on
        if (Array.isArray(targetVal) && Array.isArray(sourceVal)) {
            return (0, _lodash.isEqual)(targetVal, sourceVal) ? targetVal : sourceVal;
        }

        return sourceVal;
    };

    var preserveEquivalentArraysUnion = exports.preserveEquivalentArraysUnion = function preserveEquivalentArraysUnion(targetVal, sourceVal) {
        // By default, icepick.merge always overwrites arrays, even if they are 'equal'.
        // This differs from preserveEqualArrays in that it doesn't care about order,
        // only if the unioned list of elements is the same
        if (Array.isArray(targetVal) && Array.isArray(sourceVal)) {
            var newArray = (0, _lodash.unionWith)(targetVal, sourceVal, _lodash.isEqual);

            return (0, _lodash.isEqual)(targetVal, newArray) ? targetVal : newArray;
        }

        return sourceVal;
    };

    /**
     * If old and new state have all the same property values, return old state. Otherwise return new state.
     * @param oldState
     * @param newState
     * @returns {state}
     */
    var preserveIfEquivalent = exports.preserveIfEquivalent = function preserveIfEquivalent(oldState, newState) {
        return (0, _object.shallowEqual)(oldState, newState) ? oldState : newState;
    };

    /**
     * Reduce each action type differently. Take care to use this only at leaf-nodes of your Redux state so that
     * you can reduce atomic bits of state.
     *
     * @param {*} initialValue - starting value of state
     * @param {Object<function>} handlers - an object where each property is an action type and the value is the reducer that should handle it
     * @returns {Reducer}
     *
     * @example
     *
     * const loadingReducer = reduceByType(false, {
     *     'LOADING_START': (state, action) => true,
     *     'LOADING_END': (state, action) => false,
     * });
     *
     * loadingReducer(undefined, { type: 'UNKNOWN' }) === false;
     * loadingReducer(undefined, { type: 'LOADING_START' }) === true;
     *
     */
    var reduceByType = exports.reduceByType = function reduceByType(initialValue, handlers) {
        return function () {
            var state = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : tryFreeze(initialValue);
            var action = arguments[1];
            var context = arguments[2];

            if (handlers[action.type]) {
                return tryFreeze(handlers[action.type](state, action, context));
            }
            return state;
        };
    };

    /**
     * Map multiple action types to the same reducer. This is just syntactic sugar for reduceByType where many
     * action types are handled with the same reducer.
     *
     * @param {*} initialValue - starting value of state
     * @param {Map} handlers - a Map of string -> reducer or Array<string> -> reducer
     * @returns {Reducer}
     *
     * @example
     *
     * const busyReducer = reduceByTypes(false, [
     *     [
     *         ['LOADING_START', 'CREATE_START', 'DELETE_START', 'UPDATE_START'],
     *         (state, action) => true,
     *     ],
     *     [
     *         'LOADING_END', // simple strings are allowed
     *         (state, action) => false,
     *     ],
     * ]);
     *
     * busyReducer(undefined, { type: 'UNKNOWN' }) === false;
     * busyReducer(undefined, { type: 'LOADING_START' }) === true;
     *
     */
    var reduceByTypes = exports.reduceByTypes = function reduceByTypes(initialValue, handlers) {
        return reduceByType(initialValue, Array.from(handlers).reduce(function (expandedHandlers, _ref) {
            var _ref2 = _slicedToArray(_ref, 2),
                types = _ref2[0],
                reducer = _ref2[1];

            (0, _lodash.castArray)(types).forEach(function (type) {
                return expandedHandlers[type] = reducer;
            });
            return expandedHandlers;
        }, {}));
    };

    /**
     * Reduces state that is stored as subtrees under unique `key`s.
     *
     * NOTE: your keyGetter must know how to extract a key from every action you want to reduce.
     * This means you must either know all the actions you will want to reduce upfront, or any dynamically-added reducers
     * will need to conform to a predetermined shape that you can extract a key from.
     *
     * NOTE: Your keyGetter must also handle actions you do NOT want to reduce. Be sure you aren't assuming a payload shape.
     *
     * @param {function} keyGetter - a function that takes in an action and returns either a string key to use for the state or null to
     *                    skip reducing this action
     * @returns {function(Reducer)}
     * reducer - a reducer to be applied at the subtree level (below the key).
     *
     * @example
     *
     * // This example produces state like
     * // {
     * //     "proj/repo/2": true,
     * //     "proj/repo/4": false,
     * // }
     *
     * // Use a PR's ID as the key, and don't reduce any actions that are missing a PR
     * const reduceByPullRequest = keyBy(({ pullRequest: pr }) => isPr(pr) ? prId(pr) : null);
     *
     * // Use the MERGE_CHECK action's payload as the value for each key
     * const mergeabilityByPrReducer = keyByPullRequest((state, { type, payload: canMerge }) =>
     *     type === MERGE_CHECK
     *         ? !!canMerge
     *         : state
     * );
     *
     * mergeabilityByPrReducer(undefined, { type: 'UNKNOWN' })
     * // returns {}
     *
     * mergeabilityByPrReducer(undefined, { pullRequest: myPR })
     * // returns { myPrID: undefined }
     *
     * mergeabilityByPrReducer({ myPrID: false }, { pullRequest: myPR })
     * // returns { myPrID: false }
     *
     * mergeabilityByPrReducer(undefined, { pullRequest: yourPR, payload: true, type: MERGE_CHECK })
     * // returns { yourPrID: true }
     */
    var keyBy = exports.keyBy = function keyBy(keyGetter) {
        return function (reducer) {
            return function () {
                var state = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : (0, _icepick.freeze)({});
                var action = arguments[1];
                var context = arguments[2];

                var key = keyGetter(action);
                if (key == null) {
                    return state;
                }
                return (0, _icepick.set)(state, key, reducer(state[key], action, context));
            };
        };
    };

    /**
     * Like keyBy, but store only the latest version of the state. If the key changes, any existing state
     * under a different key is removed.
     * @param {function} keyFn - function that takes an action and returns a key to reduce under
     * @returns {function(Reducer)}
     *
     * @example
     *
     * // Use the latest commit on a branch as the key, and don't reduce actions without a latest commit
     * const reduceByHeadCommit = versionBy(({ branch }) => branch ? branch.latestCommit : null);
     *
     * // Use the BUILD_STATUS_LOADED payload
     * const branchBuildStatusReducer = reduceByHeadCommit((state, { type, payload: status }) =>
     *     type === BUILD_STATUS_LOADED
     *         ? status
     *         : state
     * );
     *
     * const badFoodBranch = { latestCommit: 'baddf00d' };
     *
     * branchBuildStatusReducer({ deadbeef: {} }, { branch: badFoodBranch })
     * // returns { baddf00d: undefined }
     *
     * branchBuildStatusReducer({ deadbeef: {} }, { branch: badFoodBranch, type: BUILD_STATUS_LOADED, payload: {} })
     * // returns { baddf00d: {} }
     */
    var versionBy = exports.versionBy = function versionBy(keyFn) {
        return function (reducer) {
            return function () {
                var state = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
                var action = arguments[1];
                var context = arguments[2];

                var key = keyFn(action);
                if (key == null) {
                    return state;
                }
                if (key in state) {
                    return (0, _icepick.set)(state, key, reducer(state[key], action, context));
                }
                // if the key doesn't match, wipe the existing state
                return _defineProperty({}, key, reducer(undefined, action, context));
            };
        };
    };

    /**
     * A helper for reducing boolean states based on action type.
     *
     * @param {string | string[]} on - type strings whose action should set the state to `true`
     * @param {string | string[]} off - type strings whose action should set the state to `false`
     * @param {string | string[]} toggle - type strings whose action should toggle the state
     * @param {string | string[]} reset - type strings whose action should set the state to `initialValue`
     * @param {*} initialValue - starting value in state
     * @returns {Reducer}
     *
     * @example
     *
     * const loadingReducer = toggleReducer({ on: LOADING, off: [LOADING_SUCCESS, LOADING_FAILURE]});
     *
     * loadingReducer(undefined, {}) // false
     * loadingReducer(true, {}) // true
     * loadingReducer(true, { type: LOADING_SUCCESS }) // false
     */
    var toggleReducer = exports.toggleReducer = function toggleReducer(_ref4) {
        var _ref4$on = _ref4.on,
            on = _ref4$on === undefined ? [] : _ref4$on,
            _ref4$off = _ref4.off,
            off = _ref4$off === undefined ? [] : _ref4$off,
            _ref4$toggle = _ref4.toggle,
            toggle = _ref4$toggle === undefined ? [] : _ref4$toggle,
            _ref4$reset = _ref4.reset,
            reset = _ref4$reset === undefined ? [] : _ref4$reset,
            _ref4$initialValue = _ref4.initialValue,
            initialValue = _ref4$initialValue === undefined ? false : _ref4$initialValue;
        return reduceByTypes(initialValue, new Map([[on, function () {
            return true;
        }], [off, function () {
            return false;
        }], [toggle, function (state) {
            return !state;
        }], [reset, function () {
            return initialValue;
        }]]));
    };

    /**
     * Handle our standard entities section of a Redux reducer - state becomes an object where keys are
     * IDs and values are the entities themselves
     *
     * @param {string | string[]} loadPageSuccess - the action type(s) on which to add a page of entities to the state
     * @param {string | string[]} loadSingleSuccess - the action type(s) on which to add a single entity to the state
     * @param idField - a property name or entity => string function to get a key for the entity
     * @returns {Reducer}
     *
     * @example
     *
     * const thingsReducer = entitiesReducer({ loadPageSuccess: LOAD_SUCCESS }, 'key');
     *
     * const things = [
     *     {
     *         key: 'red',
     *         stuff: 42,
     *     },
     *     {
     *         key: 'butterfly'
     *     }
     * ];
     *
     * thingsReducer(undefined, { type: LOAD_SUCCESS, payload: { values: things } })
     * // returns {
     * //     red: { key: 'red', stuff: 42 },
     * //     butterfly: { key: 'butterfly' },
     * // }
     */
    var entitiesReducer = exports.entitiesReducer = function entitiesReducer(_ref5, idField) {
        var _ref5$loadPageSuccess = _ref5.loadPageSuccess,
            loadPageSuccess = _ref5$loadPageSuccess === undefined ? [] : _ref5$loadPageSuccess,
            _ref5$loadSingleSucce = _ref5.loadSingleSuccess,
            loadSingleSuccess = _ref5$loadSingleSucce === undefined ? [] : _ref5$loadSingleSucce;
        return reduceByTypes({}, new Map([[loadPageSuccess, function (state, _ref6) {
            var page = _ref6.payload;
            return (0, _icepick.merge)(state, page.values.reduce(function (newEntities, entity) {
                newEntities[(0, _getId.getIdString)(idField)(entity)] = entity;
                return newEntities;
            }, {}), preserveEqualArrays);
        }], [loadSingleSuccess, function (state, _ref7) {
            var entity = _ref7.payload;
            return (0, _icepick.merge)(state, _defineProperty({}, (0, _getId.getIdString)(idField)(entity), entity), preserveEqualArrays);
        }]]));
    };

    //TODO should probably remove this and replace usage with getRequestPageReducer
    /**
     * Handle our standard paging section of a Redux reducer - state becomes an object with an array of ids and lastPageMeta
     * info.
     * @param {string | string[]} loadPageSuccess - the action type(s) on which to add a page of ids to the state
     * @param {string | function} idField - a property name or entity => string function to get a key for the entity
     * @param {boolean} append - whether to append to the existing ids or overwrite them
     * @returns {Reducer}
     *
     * @example
     *
     * const thingsReducer = _legacyPagingReducer(LOAD_SUCCESS, 'key');
     *
     * const things = [
     *     {
     *         key: 'red',
     *         stuff: 42,
     *     },
     *     {
     *         key: 'butterfly'
     *     }
     * ];
     *
     * thingsReducer(undefined, { type: LOAD_SUCCESS, payload: { values: things, anythingElse: 42 } })
     * // returns
     * // {
     * //     ids: ['red', 'butterfly'],
     * //     lastPageMeta: { anythingElse: 42 }
     * // }
     */
    var _legacyPagingReducer = exports._legacyPagingReducer = function _legacyPagingReducer() {
        var loadPageSuccess = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
        var idField = arguments[1];
        var append = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : true;
        return reduceByTypes({}, new Map([[loadPageSuccess, function (state, _ref8) {
            var page = _ref8.payload;
            return (0, _icepick.merge)(state, {
                lastPageMeta: (0, _lodash.omit)(page, 'values'),
                ids: page.values.map((0, _getId.getIdString)(idField))
            }, append ? preserveEquivalentArraysUnion : preserveEqualArrays);
        }]]));
    };

    /**
     * Build a reducer for the requests section of the store. Store whether the current request is loading,
     * metadata related to the previous request and the ids of the loaded entities.
     *
     * @param {function|string} getId - the string name of the id key of the entity or the function to generate it
     * @param {string} LOAD_PAGE - the LOAD_PAGE action type
     * @param {string} LOAD_PAGE_SUCCESS - the LOAD_PAGE_SUCCESS action type
     * @param {string} LOAD_PAGE_FAILURE - the LOAD_PAGE_FAILURE action type
     *
     * @returns {Reducer}
     */
    var getRequestPageReducer = exports.getRequestPageReducer = function getRequestPageReducer(_ref9) {
        var getId = _ref9.getId,
            LOAD_PAGE = _ref9.LOAD_PAGE,
            LOAD_PAGE_SUCCESS = _ref9.LOAD_PAGE_SUCCESS,
            LOAD_PAGE_FAILURE = _ref9.LOAD_PAGE_FAILURE;
        return (0, _redux.combineReducers)({
            loading: toggleReducer({
                on: [LOAD_PAGE],
                off: [LOAD_PAGE_SUCCESS, LOAD_PAGE_FAILURE]
            }),
            pagingMeta: reduceByType({}, _defineProperty({}, LOAD_PAGE_SUCCESS, function (state, _ref10) {
                var page = _ref10.payload;
                return (0, _lodash.omit)(page, 'values');
            })),
            ids: reduceByType([], _defineProperty({}, LOAD_PAGE_SUCCESS, function (state, _ref11) {
                var page = _ref11.payload;
                return state.concat(page.values.map((0, _getId.getIdString)(getId)));
            }))
        });
    };

    /**
     * Build a reducer for the requests section of the store. Store whether the current request is loading
     * and metadata related to the previous request
     *
     * @param {string} REQUEST - the REQUEST action type
     * @param {string} REQUEST_SUCCESS - the REQUEST_SUCCESS action type
     * @param {string} REQUEST_FAILURE - the REQUEST_FAILURE action type
     *
     * @returns {Reducer}
     */
    var getRequestReducer = exports.getRequestReducer = function getRequestReducer(_ref12) {
        var REQUEST = _ref12.REQUEST,
            REQUEST_SUCCESS = _ref12.REQUEST_SUCCESS,
            REQUEST_FAILURE = _ref12.REQUEST_FAILURE;
        return (0, _redux.combineReducers)({
            loading: toggleReducer({
                on: [REQUEST],
                off: [REQUEST_SUCCESS, REQUEST_FAILURE]
            }),
            response: reduceByType({}, _defineProperty({}, REQUEST_SUCCESS, function (state, _ref13) {
                var payload = _ref13.payload;
                return payload;
            }))
        });
    };

    /**
     * Build the reducers for page requests for a given collection of request types and key props
     * @param {object} requestTypeKeyProps - The map of request types to the payload properties used to key the request
     * @param {function} getId - the id getter for this entity type
     * @param {string} LOAD_PAGE - LOAD_PAGE action type
     * @param {string} LOAD_PAGE_FAILURE - LOAD_PAGE_FAILURE action type
     * @param {string} LOAD_PAGE_SUCCESS - LOAD_PAGE_SUCCESS action type
     * @returns {Reducer}
     */
    var getRequestPageReducersByType = exports.getRequestPageReducersByType = function getRequestPageReducersByType(_ref14) {
        var requestTypeKeyProps = _ref14.requestTypeKeyProps,
            getId = _ref14.getId,
            LOAD_PAGE = _ref14.LOAD_PAGE,
            LOAD_PAGE_FAILURE = _ref14.LOAD_PAGE_FAILURE,
            LOAD_PAGE_SUCCESS = _ref14.LOAD_PAGE_SUCCESS;

        var reducer = (0, _lodash.mapValues)(requestTypeKeyProps, function (props, requestType) {
            return composeReducers(reduceIf(isRequestType(requestType)), keyByRequestProps(props), getRequestPageReducer({
                getId: getId,
                LOAD_PAGE: LOAD_PAGE,
                LOAD_PAGE_FAILURE: LOAD_PAGE_FAILURE,
                LOAD_PAGE_SUCCESS: LOAD_PAGE_SUCCESS
            }));
        });

        return composeReducers(reduceIf(actionTypeIn(LOAD_PAGE, LOAD_PAGE_FAILURE, LOAD_PAGE_SUCCESS)), reducer);
    };

    var getRequestReducerByType = exports.getRequestReducerByType = function getRequestReducerByType(_ref15) {
        var requestTypeKeyProps = _ref15.requestTypeKeyProps,
            REQUEST = _ref15.REQUEST,
            REQUEST_FAILURE = _ref15.REQUEST_FAILURE,
            REQUEST_SUCCESS = _ref15.REQUEST_SUCCESS;

        var reducer = (0, _lodash.mapValues)(requestTypeKeyProps, function (props, requestType) {
            return composeReducers(reduceIf(isRequestType(requestType)), keyByRequestProps(props), getRequestReducer({
                REQUEST: REQUEST,
                REQUEST_FAILURE: REQUEST_FAILURE,
                REQUEST_SUCCESS: REQUEST_SUCCESS
            }));
        });

        return composeReducers(reduceIf(actionTypeIn(REQUEST, REQUEST_FAILURE, REQUEST_SUCCESS)), reducer);
    };

    /**
     * Handle a list of error messages on failure and reset them on success. Expects the action shapes from
     * rest-actor.js::restActorForType (e.g. a failure has `payload.response.errors`)
     *
     * @param {*} initialValue - whatever the default state should be (you might want this to be null or an empty array)
     * @param {string | string[]} successAction - the action that will reset to the default state
     * @param {string | string[]} failureAction - the action which will have a payload response with errors
     * @returns {Reducer}
     *
     * @example
     *
     * const reducer = errorsReducer([], SUCCESS, FAILURE);
     *
     * reducer('blah', { type: SUCCESS }) // returns []
     *
     * const response = { errors: [{ message: 'Only time can heal what reason cannot.' }] }
     *
     * reducer('blah', { type: FAILURE, payload: { response } })
     * // returns [{ message: 'Only time can heal what reason cannot.' }]
     */
    var errorsReducer = exports.errorsReducer = function errorsReducer(initialValue) {
        var successAction = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : [];
        var failureAction = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : [];

        var errorsReducers = new Map([[successAction, function () {
            return initialValue;
        }], [failureAction, function (state, _ref16) {
            var response = _ref16.payload.response;
            return response && response.errors || [{ message: 'An unknown error occurred.' }];
        }]]);

        return reduceByTypes(initialValue, errorsReducers);
    };

    /**
     * Execute a given reducer only if the action passes some predicate
     * @param {function(Action)} predicate - The function which determines whether the reducer should be run
     * @param {*} initialState - starting value of the state
     *
     * @returns {function(Reducer)}
     */
    var reduceIf = exports.reduceIf = function reduceIf(predicate) {
        var initialState = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
        return function (reducer) {
            return function () {
                var state = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : initialState;
                var action = arguments[1];
                return predicate(action) ? reducer(state, action) : state;
            };
        };
    };

    /**
     * Chain together a collection of higher order reducers, terminated by a standard reducer
     * Support shorthand for `combineReducers` by passing an object literal as the last reducer
     *
     * @param {...(function(Reducer)|Reducer|Object)} reducers - The collection of reducers to compose.
     *
     * @returns {Reducer}
     */
    var composeReducers = exports.composeReducers = function composeReducers() {
        for (var _len = arguments.length, reducers = Array(_len), _key = 0; _key < _len; _key++) {
            reducers[_key] = arguments[_key];
        }

        var lastReducer = reducers.pop();

        // eslint-disable-next-line lodash/consistent-compose
        return (0, _lodash.flowRight)(reducers)((0, _lodash.isPlainObject)(lastReducer) ? (0, _redux.combineReducers)(lastReducer) : lastReducer);
    };

    /**
     * Return the payload from the triggering action for a RestActor for either the triggering action
     * or the success/failure actions
     *
     * @param {Action} action - The action to get the payload from
     *
     * @returns {*}
     */
    var getRequestPayload = exports.getRequestPayload = function getRequestPayload(action) {
        return (0, _lodash.get)(action, 'meta.originalAction.payload', action.payload);
    };

    /**
     * Gets a key from a property of the supplied action payload.
     * Prefixes the key with the property name and uses `UNKEYED` if the property is undefined
     *
     * @param {string|string[]} prop - the name of the property to use,
     *      or an array with the prop name and a value getter function
     * @param {object} payload - the action payload to get the value from
     *
     * @returns {string}
     */
    var getKeyFromProp = exports.getKeyFromProp = function getKeyFromProp(prop, payload) {
        var _castArray = (0, _lodash.castArray)(prop),
            _castArray2 = _slicedToArray(_castArray, 2),
            propName = _castArray2[0],
            valueFn = _castArray2[1];

        var propVal = (0, _lodash.get)(payload, propName);

        if (valueFn && propVal) {
            propVal = valueFn(propVal);
        }

        if (!(0, _lodash.isString)(propVal)) {
            propVal = JSON.stringify(propVal);
        }

        return propName + ':' + (propVal || UNKEYED);
    };

    /**
     * Key the reducer by a property from the request payload of a `RestActor` related action
     * (e.g LOAD, LOAD_SUCCESS, LOAD_FAILURE)
     *
     * @param {string|string[]} prop - the name of the property to use,
     *      or an array with the prop name and a value getter function
     *
     * @returns {function(Reducer)}
     */
    var keyByRequestProp = exports.keyByRequestProp = function keyByRequestProp(prop) {
        return keyBy(function (action) {
            var payload = getRequestPayload(action);

            return getKeyFromProp(prop, payload);
        });
    };

    /**
     * Build a nested key hierarchy based on a number of properties from the request payload
     * of a `RestActor` related action (e.g LOAD, LOAD_SUCCESS, LOAD_FAILURE)
     *
     * @param {string[]} props - an array with the names of the property to use,
     *      or arrays with the prop name and a value getter function
     *
     * @returns {function(Reducer)}
     */
    var keyByRequestProps = exports.keyByRequestProps = function keyByRequestProps(props) {
        return (
            // eslint-disable-next-line lodash/consistent-compose
            _lodash.flowRight.apply(undefined, _toConsumableArray(props.map(keyByRequestProp)))
        );
    };

    /**
     * Check if `requestType` property of the original request action matches the one supplied
     * @param {string} requestType - The request type to match
     * @returns {function(*=): boolean}
     */
    var isRequestType = exports.isRequestType = function isRequestType(requestType) {
        return function (action) {
            var payload = getRequestPayload(action);

            return (0, _lodash.get)(payload, 'requestType') === requestType;
        };
    };

    /**
     * Check if the action type for the action is in the supplied list
     * @param {...string} actionTypes
     * @returns {function(*): boolean}
     */
    var actionTypeIn = exports.actionTypeIn = function actionTypeIn() {
        for (var _len2 = arguments.length, actionTypes = Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
            actionTypes[_key2] = arguments[_key2];
        }

        return function (action) {
            return actionTypes.includes(action.type);
        };
    };

    /**
     * For a single entity request, assume the payload is an entity and key by its ID
     * @param {function} getId - the id getter for this entity type
     * @returns {function(Reducer)}
     */
    var keyByPayloadId = exports.keyByPayloadId = function keyByPayloadId(getId) {
        return keyBy(function (action) {
            return getId(action.payload);
        });
    };

    /**
     * Build the reducer for a single entity request (usually load or save a single entity)
     * @param {string|string[]} start - The action type(s) that indicate the start of the request
     * @param {string|string[]} end - The action type(s) that indicate the end of the request
     * @param {function} getId - the id getter for this entity type
     * @returns {Reducer}
     */
    var singleEntityRequestReducer = exports.singleEntityRequestReducer = function singleEntityRequestReducer(_ref17) {
        var start = _ref17.start,
            end = _ref17.end,
            getId = _ref17.getId;
        return composeReducers(reduceIf(actionTypeIn.apply(undefined, _toConsumableArray((0, _lodash.castArray)(start)).concat(_toConsumableArray((0, _lodash.castArray)(end))))), keyByPayloadId(getId), {
            loading: toggleReducer({
                on: start,
                off: end
            })
        });
    };
});