define('bitbucket/internal/next/feature/changes/changes-tree/utils/paths-sorting-comparators', ['exports', 'bitbucket/internal/widget/tree-view/enums'], function (exports, _enums) {
    'use strict';

    Object.defineProperty(exports, "__esModule", {
        value: true
    });
    exports.filePathsComparator = filePathsComparator;
    exports.branchPathsComparator = branchPathsComparator;


    var SYMBOL_PREFIX_FILE_REGEXP = /^[\.\-_]/;

    var isDotName = function isDotName(filename) {
        return SYMBOL_PREFIX_FILE_REGEXP.test(filename);
    };

    // Intl.Collator is faster than String.prototype.localeCompare
    // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/localeCompare#Performance
    var naturalLocaleCompare = new Intl.Collator(undefined, {
        numeric: true
    }).compare;

    function filePathsComparator(pathA, pathB) {
        var partsA = pathA.split(_enums.PATH_SEPARATOR);
        var partsB = pathB.split(_enums.PATH_SEPARATOR);

        var partsASize = partsA.length;
        var partsBSize = partsB.length;

        for (var i = 0, limit = Math.max(partsASize, partsBSize); i < limit; i++) {
            if (!(i in partsA)) {
                return +1;
            }

            if (!(i in partsB)) {
                return -1;
            }

            var nameA = partsA[i];
            var nameB = partsB[i];

            // A is a directory but B is a file
            if (partsA[i + 1] && !partsB[i + 1]) {
                return -1;
            }

            // A is a file but B is a directory
            if (!partsA[i + 1] && partsB[i + 1]) {
                return +1;
            }

            // A is a "dot" name but B is regular name
            if (isDotName(nameA) && !isDotName(nameB)) {
                return -1;
            }

            // A is regular name but B is a "dot" name
            if (!isDotName(nameA) && isDotName(nameB)) {
                return +1;
            }

            var compareResult = naturalLocaleCompare(nameA, nameB);

            // A and B are different
            if (compareResult !== 0) {
                return compareResult;
            }
        }
    }

    function dirPathsComparator(pathA, pathB) {
        var partsA = pathA.split(_enums.PATH_SEPARATOR);
        var partsB = pathB.split(_enums.PATH_SEPARATOR);

        for (var i = 0, limit = Math.max(partsA.length, partsB.length); i < limit; i++) {
            if (!(i in partsA)) {
                return +1;
            }

            if (!(i in partsB)) {
                return -1;
            }

            var nameA = partsA[i];
            var nameB = partsB[i];

            // A is a "dot" name but B is regular name
            if (isDotName(nameA) && !isDotName(nameB)) {
                return -1;
            }

            // A is regular name but B is a "dot" name
            if (!isDotName(nameA) && isDotName(nameB)) {
                return +1;
            }

            var compareResult = naturalLocaleCompare(nameA, nameB);

            // A and B are different
            if (compareResult !== 0) {
                return compareResult;
            }
        }
    }

    function branchPathsComparator(pathA, typeA, pathB, typeB) {
        // A is a directory and B is a file
        if (typeA === _enums.NODE_TYPE_DIRECTORY && typeB !== _enums.NODE_TYPE_DIRECTORY) {
            return -1;
        }

        // B is a directory and A is a file
        if (typeA !== _enums.NODE_TYPE_DIRECTORY && typeB === _enums.NODE_TYPE_DIRECTORY) {
            return +1;
        }

        // A and B are same type
        if (typeA === typeB) {
            // A is a "dot" name and B is a regular name
            if (isDotName(pathA) && !isDotName(pathB)) {
                return -1;
            }

            // A is a regular name and B is a "dot" name
            if (!isDotName(pathA) && isDotName(pathB)) {
                return +1;
            }
        }

        return dirPathsComparator(pathA, pathB);
    }
});