import validator from 'bitbucket/internal/json-validation';
import _ from 'lodash';

/**
 * Any person. Could be a user of the application, a committer to a repository, etc.
 *
 * @class Person
 * @property {string?} email_address - email for the user. Usually only appears on the current user
 * @property {string} name - @username for the user
 */
const person = {
    email_address: 'string?',
    name: 'string'
};

/**
 * A user type
 * @enum {string}
 */
const UserType = {
    NORMAL : 'NORMAL',
    SERVICE: 'SERVICE'
};

/**
 * A project type
 * @enum {string}
 */
const ProjectType = {
    NORMAL: 'NORMAL',
    PERSONAL: 'PERSONAL'
};

/**
 * The SCM id
 * @enum {string}
 */
const scmId = {
    GIT: 'git',
    HG: 'hg'
};

/**
 * A person who is a user of this application.
 *
 * Note that the following properties will be used by frontend components directly.
 * In order to satisfy some SPIs, you may need additional properties on your model.
 * For example, Server needs an avatarUrl property to accurately provide avatar urls in
 * bitbucket/internal/urls
 *
 * @class User
 * @extends Person
 * @param {boolean} is_active - whether the user still has an active account
 * @param {string} display_name - the name to show in UI for the user
 * @param {boolean} is_admin - whether the user is an instance-wide admin. Can be used to
 *                            provide extra links/actions that are only available to instance
 *                            admins (e.g., edit other users, etc). Of course checks on the
 *                            backend should stop non-admins from making use of those links
 *                            regardless of this value.
 * @param {UserType} type - is this a real person or service user?
 * @param {string} avatar_url - The URL to the avatar of this user.
 */
const user = _.extend({}, person, {
    is_active: 'boolean',
    display_name: 'string',
    is_admin: 'boolean',
    type: validator.asEnum('UserType', UserType),
    avatar_url: 'string'
});

// Currently (as of 2015-07-07) only Server has the concept of a Project
const project = {
    id: 'string',
    name: 'string',
    key: 'string',
    description: 'string?',
    type: validator.asEnum('ProjectType', ProjectType),
    owner : validator.nullable(user),
    is_public: 'boolean'
};

const repository = {
    id: 'string',
    name: 'string',
    slug: 'string',
    is_forkable : 'boolean',
    is_public : 'boolean',
    origin: validator.nullable(validator.recurse(() => repository)),
    scm_id: validator.asEnum('scmId', scmId)
};

/**
 * The type of a ref
 * @enum {string}
 */
const RefType = {
    TAG: 'tag',
    BRANCH: 'branch',
    COMMIT: 'commit'
};

const ref = {
    id : 'string',
    display_id : 'string',
    type : validator.asEnum('RefType', RefType),
    is_default : 'boolean',
    latest_commit : 'string',
    repository
};

/**
 * Pull request participant's role
 * @enum {string}
 */
const ParticipantRole = {
    AUTHOR: 'author',
    REVIEWER: 'reviewer',
    PARTICIPANT: 'participant'
};

/**
 * Pull request Participant's approval state of the pull request
 * @enum {string}
 */
const ApprovalState = {
    APPROVED: 'APPROVED',
    REVIEWED: 'REVIEWED',
    UNREVIEWED: 'UNREVIEWED'
};

/**
 * Pull request state
 * @enum {string}
 */
const PullRequestState = {
    OPEN: 'OPEN',
    MERGED: 'MERGED',
    DECLINED: 'DECLINED'
};

const abstractParticipant = {
    user,
    role: validator.asEnum('Role', ParticipantRole),
    state: validator.asEnum('ApprovalState', ApprovalState)
};
const author = _.extend({}, abstractParticipant, { role: validator.strictEqual(ParticipantRole.AUTHOR) });
const reviewer = _.extend({}, abstractParticipant, { role: validator.strictEqual(ParticipantRole.REVIEWER) });
const participant = _.extend({}, abstractParticipant, { role: validator.strictEqual(ParticipantRole.PARTICIPANT) });

const pull_request = {
    author: author,
    description: 'string?',
    description_as_html: 'string?',
    created_date: 'string', // should be an ISO-8601 time string
    from_ref: ref,
    id: 'string?',
    participants: [ participant ],
    reviewers: [ reviewer ],
    state: validator.asEnum('PullRequestState', PullRequestState),
    title: 'string?',
    to_ref: ref,
    updated_date: 'string', // should be an ISO-8601 time string
    version: 'number?'
};

const comment = {}; // TODO

const task = {}; // TODO

// a change between two commits, including file diffs.
const changeset = {}; // TODO

// information about a file - e.g. filename, binary vs text, etc.
const file_metadata = {}; // TODO

// the source of a text file.
const file_text = {}; // TODO

const permission = {
    name: validator.asEnum('permission', {
        KNOWN_USER: 'KNOWN_USER', // == LICENSED_USER on Server -> any logged-in user
        REPO_READ: 'REPO_READ', // can view the given repo
        REPO_WRITE: 'REPO_WRITE', // can push to the given repo
        REPO_ADMIN: 'REPO_ADMIN', // can administrate the given repo
        SYS_ADMIN: 'SYS_ADMIN' // on Cloud, I suspect this is just the Bitbucket team.
    }),
    project_id: 'string?', // in Server this is a number.
    project_key: 'string?', // the URL identifier for the project
    repository_id: 'string?', // in Server this is a number
    repository_slug: 'string?' // the URL identifier for the repository. If specified, project_key must also be specified.
};

export default {

    // Enums
    ApprovalState,
    ProjectType,
    PullRequestState,
    RefType,
    ParticipantRole,
    scmId,
    UserRole: ParticipantRole, // deprecated in 1.3.0. Used in Stash. But usage was never released. Please remove in Sep 2015 - no breaking change needed.
    UserType,

    // Models
    changeset,
    comment,
    file_metadata,
    file_text,
    permission,
    person,
    project,
    pull_request,
    ref,
    repository,
    task,
    user
};
