import FilterBar from 'bitbucket/internal/filter-bar';
import AJS from 'aui';
import $ from 'jquery';

QUnit.module('bitbucket-filter-bar', {
    afterEach: function () {
        // since using select2 commands can open the select and place it outside
        // of the fixture, remove select2 dropdowns when we're done with them.
        $('body > .select2-drop').remove();
    }
});

class DummyDataProvider {
    constructor() {
        this.filter = {};
        this.setFilter = sinon.spy();
        this.reset = sinon.spy();
    }

    fetchNext() {
        this.reachedEnd = true;
        return {
            then: sinon.spy(function(done) {
                done([{
                    id: 'item-one',
                    text: 'item one'
                }]);
            })
        };
    }
}

function validFilterBar(initialValues) {
    var provider = new DummyDataProvider();
    var bar = new FilterBar(document.querySelector('#qunit-fixture'), {
        id: 'filter-bar',
        filters: [{
            id: 'filter-one',
            text: 'filter one',
            type: FilterBar.Types.TOGGLE,
            value: initialValues && initialValues['filter-one']
        }, {
            id: 'filter-two',
            text: 'filter two',
            type: FilterBar.Types.SELECT,
            menu: {
                minimumInputLength: 0, // force a search box
                dropdownCssClass: 'filter-two-dropdown',
                items : [{
                    id : 'item-one',
                    text: 'item one'
                }, {
                    id : 'item-two',
                    text: 'item two'
                }]
            },
            value: initialValues && initialValues['filter-two']
        }, {
            id: 'filter-three',
            text: 'filter three',
            type: FilterBar.Types.SELECT,
            searchPlaceholder: 'filter-three-search-placeholder',
            menu: {
                dataProvider: provider,
                placeholder: 'placeholder',
                dropdownCssClass: 'filter-three-dropdown',
                initSelection: function (element, callback) {
                    var id = $(element).val();
                    new DummyDataProvider().fetchNext().then(function (items) {
                        callback(items[0]);
                    });
                }
            },
            value: initialValues && initialValues['filter-three']
        }]
    });
    bar.__provider = provider;
    return bar;
}

QUnit.test('validates filters - valid', (assert) => {
    validFilterBar();
    assert.ok(true, 'No exceptions thrown creating a valid filter bar.');
});


QUnit.test('validates filters - invalid', (assert) => {
    function test(filter) {
        assert.throws(function() {
            new FilterBar(document.querySelector('#qunit-fixture'), {
                id: 'filter-bar',
                filters: [filter]
            });
        });
    }
    test({
        text: 'filter one',
        type: FilterBar.Types.TOGGLE
    });
    test({
        id: 'filter-one',
        type: FilterBar.Types.TOGGLE
    });
    test({
        id: 'filter-one',
        text: 'filter one'
    });
    test({
        id: 'filter-one',
        text: 'filter one',
        type: FilterBar.Types.SELECT
    });
    test({
        id: 'filter-one',
        text: 'filter one',
        type: FilterBar.Types.SELECT,
        menu: {}
    });
    test({
        id: 'filter-one',
        text: 'filter one',
        type: FilterBar.Types.SELECT,
        menu: {
            dataProvider: new DummyDataProvider()
        }
    });
});

QUnit.test('creates filters', function(assert) {
    var filterBar = validFilterBar();
    var bar = document.querySelector('#filter-bar');
    assert.ok(!!bar, 'Bar was added to DOM');
    var toggle = bar.querySelector('#filter-one');
    assert.ok(!!toggle, 'Toggle was added to DOM');
    var staticSelect = bar.querySelector('#filter-two');
    assert.ok(!!staticSelect, 'Static select was added to DOM');
    var ajaxSelect = bar.querySelector('#filter-three');
    assert.ok(!!ajaxSelect, 'AJAX select was added to DOM');
});

QUnit.test('change triggers and updates state', function(assert) {
    var filterBar = validFilterBar();
    var changeSpy = sinon.spy();
    filterBar.on('change', changeSpy);

    assert.deepEqual(filterBar.state, {
        'filter-one': false,
        'filter-two': 'item-one', // first option selected by default in static selects
        'filter-three': ''
    });
    assert.equal(changeSpy.callCount, 0, 'No change occurred yet.');

    var $ajaxFilter = $('#filter-three');
    assert.equal(filterBar.__provider.setFilter.callCount, 0, 'no filter set initially');
    assert.equal(filterBar.__provider.reset.callCount, 0, 'no initial reset');

    $ajaxFilter.select2('search', 'item');
    assert.equal(filterBar.__provider.setFilter.callCount, 2, 'setFilter called on open and when term changed (facepalm)');
    assert.equal(filterBar.__provider.reset.callCount, 2, 'reset called on open and when term changed (facepalm)');

    $ajaxFilter.select2('val', 'item-one', true);

    assert.equal($ajaxFilter.val(), 'item-one', 'Correct value.');
    assert.equal(changeSpy.callCount, 1, 'Change triggered.');
    assert.deepEqual(filterBar.state, {
        'filter-one': false,
        'filter-two': 'item-one',
        'filter-three': 'item-one'
    });

    var $toggle = $('#filter-one');
    $toggle.click();
    assert.equal(changeSpy.callCount, 2, 'Change triggered.');
    assert.deepEqual(filterBar.state, {
        'filter-one': true,
        'filter-two': 'item-one',
        'filter-three': 'item-one'
    });
});

QUnit.test('initial values are respected', function(assert) {
    var expectedState = {
        'filter-one': true,
        'filter-two': 'item-two',
        'filter-three': 'item-one'
    };
    var filterBar = validFilterBar(expectedState);
    var state = filterBar.state;
    assert.strictEqual(state['filter-one'], expectedState['filter-one']);
    assert.strictEqual(state['filter-two'], expectedState['filter-two']);
    assert.strictEqual(state['filter-three'], expectedState['filter-three']);
    assert.deepEqual(state, expectedState);
});

QUnit.test('components in filter bars do not mess with eachother', function(assert) {
    let filterBar = validFilterBar();

    // first check that we have multiple search fields:
    assert.ok($('.select2-search > input').length > 1, 'more than 1 search field');

    let filter3 = filterBar.options.filters.filter((item) => item.id === 'filter-three')[0];
    // trigger the select2-open on filter-three (the async select with a search)
    // and verify that only a single search input has its placeholder set
    filterBar._$filterBar.find(`#${filter3.id}`).trigger('select2-open');
    assert.equal($(`input[placeholder="${filter3.searchPlaceholder}"]`).length, 1, "should be only one input with a given placeholder");

});
