define('bitbucket/internal/feature/file-content/file-blame/blame-gutter', [
    'jquery',
    'lodash',
    'bitbucket/util/events',
    'bitbucket/util/state',
    'bitbucket/internal/util/css',
    'bitbucket/internal/util/object',
    'bitbucket/internal/widget'
], function (
    $,
    _,
    events,
    pageStateApi,
    css,
    obj,
    Widget
) {
    'use strict';

    function createBlameElementCache(blames) {
        var repo = pageStateApi.getRepository();
        var filePath = pageStateApi.getFilePath().components.join('/');
        var blameStats = blames.map(function(blameSpan) {
            var templateData = {
                spannedLines : blameSpan.spannedLines,
                filePath: filePath,
                repository: repo,
                commit: _.extend({
                    id: blameSpan.commitHash,
                    displayId: blameSpan.displayCommitHash
                }, blameSpan)
            };
            var firstEl = $(bitbucket.internal.feature.fileContent.fileBlameGutterDetailed(templateData))[0];
            if (blameSpan.spannedLines === 1) {
                return {
                    commitId: blameSpan.commitHash,
                    els: [firstEl]
                };
            }

            var restEl = $(bitbucket.internal.feature.fileContent.fileBlameGutterSpan(templateData))[0];
            return {
                commitId: blameSpan.commitHash,
                els: [firstEl, restEl].concat(_.times(blameSpan.spannedLines - 2, function () {
                    return restEl.cloneNode(true);
                }))
            };
        });
        var all = _.chain(blameStats).pluck('els').flatten().value();

        var stat = {
            byLine: {},
            byCommitId: {},
            all: all
        };

        _.transform(stat.all, function(byLine, blameEl, index) {
            byLine[index + 1] = blameEl;
        }, stat.byLine);
        _.transform(blameStats, function(byCommitId, stat) {
            if (byCommitId[stat.commitId]) {
                byCommitId[stat.commitId] = byCommitId[stat.commitId].concat(stat.els);
            } else {
                byCommitId[stat.commitId] = stat.els;
            }
        }, stat.byCommitId);
        return stat;
    }

    var BLAME_GUTTER_ID = 'blame';

    function getSetGutterMarkerArgs(blameElementsByLine, change) {
        var setGutterMarkerArgs = [];
        return change.eachLine(function(lineInfo) {
            var blameEl = blameElementsByLine[lineInfo.lineNumber];
            setGutterMarkerArgs.push([ lineInfo.handles.SOURCE, BLAME_GUTTER_ID, blameEl ]);
        }).then(function() { return setGutterMarkerArgs; });
    }

    function BlameGutter(sourceView, blameSource) {
        Widget.call(this);
        this._enabled = false;
        this._sourceView = sourceView;
        this._blameSource = blameSource;
        this._pendingChanges = [];

        var self = this;
        sourceView.on('destroy', function() {
            self.destroy();
        });
        sourceView.on('change', function(change) {
            if (self._pendingChanges) {
                self._pendingChanges.push(change);
            } else {
                self._fillForChange(change);
            }
        });

        this._sourceView.addContainerClass('blame-disabled');
        sourceView.registerGutter(BLAME_GUTTER_ID, { weight: 0 });
    }
    obj.inherits(BlameGutter, Widget);

    BlameGutter.prototype.setEnabled = function(shouldEnable) {
        shouldEnable = Boolean(shouldEnable);
        var whenChanged;
        if (this._enabled !== shouldEnable) {
            this._enabled = shouldEnable;
            if (this._enabled === shouldEnable) { // event listener didn't call setEnabled.
                if (shouldEnable) {
                    this._sourceView.removeContainerClass('blame-disabled');
                    whenChanged = this._fillGutter();
                } else {
                    this._sourceView.addContainerClass('blame-disabled');
                    whenChanged = $.Deferred().resolve();
                }
                var self = this;
                whenChanged.done(function () {
                    events.trigger('bitbucket.internal.feature.fileContent.fileBlameExpandedStateChanged', null, self._enabled);
                });
            }
        }
        return whenChanged || $.Deferred().resolve();
    };

    BlameGutter.prototype._addHoverBehavior = function() {
        var sourceView = this._sourceView;
        var $allElements = $(this._blameElCache.all);
        var $byCommitId = _.transform(this._blameElCache.byCommitId, function($byCommitId, els, commitId) {
            $byCommitId[commitId] = $(els);
        }, {});
        var unhover;
        var hoveredCommitId;
        var unhoverTimeout;


        function mouseEnter(e) {
            if (e.target !== this) {
                return;
            }
            var commitId = this.getAttribute('data-commitid');
            clearTimeout(unhoverTimeout);

            if (hoveredCommitId === commitId) {
                return;
            }

            if (unhover) {
                unhover();
            }

            hoveredCommitId = commitId;
            var $newHovered = $byCommitId[commitId];
            var unstyle;
            // At some point the number of elements out of the DOM is so great it's actually slower to change them than to change global styles.
            // This especially affects IE, Firefox slightly, and Chrome almost not at all. But the result is that for very large
            // blames, we add rules to style them, instead of adding style classes
            if ($newHovered.length < 500) {
                $newHovered.addClass('commitid-hovered');
                unstyle = $newHovered.removeClass.bind($newHovered, 'commitid-hovered');
            } else {
                unstyle = css.appendRule('.blame-row.blame-row[data-commitid="' + commitId + '"] {' +
                    // Change with @primaryHighlightColor and @primaryHighlight
                    'background-color: #f5f5f5;' +
                    'border-right-color: #3b73af;' +
                    '}');
            }
            unhover = function() {
                unstyle();
                hoveredCommitId = null;
                unhover = null;
            };
        }

        function mouseLeave(e) {
            if (e.target !== this) {
                return;
            }
            if (unhover) {
                unhoverTimeout = setTimeout(function() {
                    if (unhover) {
                        unhover();
                    }
                }, 100);
            }
        }

        //$allElements.hover(mouseEnter, mouseLeave);
        // jQuery event handling invalidates layout due to copying of some properties on the event
        // So we're resorting to native handling.
        // Turns a 40ms frame into a 30ms one due to not having to do weird requests for SVG data URIs.
        this._blameElCache.all.forEach(function(el) {
            el.addEventListener('mouseenter', mouseEnter);
            el.addEventListener('mouseleave', mouseLeave);
        });

        this._addDestroyable(function() {
            if (unhover) { // must call this to remove added CSS rules.
                unhover();
            }
        });
    };

    /**
     * Fill the gutter for a given change.
     *
     * @param {SourceViewChange} change
     * @private
     */
    BlameGutter.prototype._fillForChange = function(change) {
        var self = this;
        getSetGutterMarkerArgs(self._blameElCache.byLine, change).done(function(setGutterMarkerArgs) {
            self._sourceView.operation(function() {
                setGutterMarkerArgs.forEach(function(args) {
                    self._sourceView.setGutterMarker.apply(self._sourceView, args);
                });
            });
        });
    };

    /**
     * Fill the gutter the first time it is shown. Requires requesting the blame information for all existing changes.
     *
     * @private
     */
    BlameGutter.prototype._fillGutter = function() {
        if (this._gutterFilled) {
            return $.Deferred().resolve();
        }
        this._gutterFilled = true;

        var self = this;
        return this._blameSource.get().then(function(blames) {
            self._blameElCache = createBlameElementCache(blames);
            self._addHoverBehavior();
            self._sourceView.operation(function() {
                self._pendingChanges.forEach(self._fillForChange.bind(self));
            });
            self._pendingChanges = null;
        });
    };

    return BlameGutter;
});
