define('bitbucket/internal/util/shortcuts', [
    'bacon',
    'bitbucket/internal/util/deprecation',
    'bitbucket/internal/widget/keyboard-shortcuts',
    'exports'
], function(
    Bacon,
    deprecate,
    keyboardShortcuts,
    exports
) {

    //TODO STASHDEV-8944: Replace this with a method for keyboard setup that doesn't rely on whenIType running an `evaluate`

    'use strict';

    var baconStreamInfos = {};

    /**
     * @typedef  {Object} StreamInfo
     * @property {List<WhenIType>} initialWhenITypes - A list of WhenITypes that are added when the stream is bound.
     * @property {Function|null}   sink              - The Sink function if the stream is currently bound, could be
     *                                                 null if the stream is not currently bound.
     * @property {Bacon}           stream            - The Bacon stream.
     */

    /**
     * Gets the Stream information
     *
     * @param {string} name - name of the keyboard shortcut.
     * @returns {StreamInfo}
     */
    function getStreamInfo(name) {
        if (!baconStreamInfos.hasOwnProperty(name)) {
            baconStreamInfos[name] = {
                // We have to do a bit of a dance with _initialWhenITypes, because the binder function wont be evaluated
                // until the stream is onValue()ed.
                analyticsBound: false,
                _sink: null,

                // The bacon sink is for exposing the internal sink function to the outside world from
                // baconjs' fromBinder method.
                // It is needed by whenIType.execute() to register the keyboard event as the source of the event stream.
                baconSink: function (e) {
                    if (this._sink) {
                        this._sink(e);
                    }
                },
                stream: Bacon.fromBinder(function (sink) {
                    baconStreamInfos[name]._sink = sink;
                    return function() {
                        baconStreamInfos[name]._sink = null;
                    };
                })
            };
        }
        return baconStreamInfos[name];
    }

    /**
     * Setups a new Keyboard Shortcut with a WhenIType
     *
     * Can be called multiple times for the same shortcut if the shortcut is registered multiple times.
     *
     * @param {string}    name      - Name of the keyboard shortcut
     * @param {WhenIType} whenIType - WhenIType object that has been bound
     * @param {string}    keys      - Keys that this keyboard shortcut uses
     */
    function setup(name, whenIType, keys) {
        var streamInfo = getStreamInfo(name);
        whenIType.execute(streamInfo.baconSink.bind(streamInfo));

        // as part of STASHDEV-8944 we should remove the concept of shortcut contexts that way we will be able to do analytics here.
        if (!streamInfo.analyticsBound) {
            streamInfo.analyticsBound = true;
            keyboardShortcuts.bindKeyboardShortcutAnalytics(name, baconStreamInfos[name].stream);
        }
        deprecate.triggerDeprecated('bitbucket.internal.keyboard.shortcuts.' + name, whenIType, keys, null, '3.9', '4.0');
    }

    /**
     * Binds a keyboard shortcut
     *
     * @param {string}          name     - Name of the keyboard shortcut.
     * @param {Function<Event>} callback - A function to call when the keyboard shortcut is typed.
     * @return {Function}                - A function you can call to unbind the callback
     */
    function bind(name, callback) {
        return getStreamInfo(name).stream.onValue(callback);
    }

    exports.setup = setup;
    exports.bind = bind;
});
