package com.atlassian.bitbucket.ssh.command;

import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableMap;

import javax.annotation.Nonnull;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Map;

import static java.util.Objects.requireNonNull;

/**
 * @since 5.12
 */
public class SshCommandContext {

    private final String command;
    private final Map<String, String> environment;
    private final OutputStream stderr;
    private final InputStream stdin;
    private final OutputStream stdout;

    private SshCommandContext(Builder builder) {
        command = builder.command;
        environment = builder.environment.build();
        stderr = builder.stderr;
        stdin = builder.stdin;
        stdout = builder.stdout;
    }

    @Nonnull
    public String getCommand() {
        return command;
    }

    @Nonnull
    public Map<String, String> getEnvironment() {
        return environment;
    }

    @Nonnull
    public OutputStream getStderr() {
        return stderr;
    }

    @Nonnull
    public InputStream getStdin() {
        return stdin;
    }

    @Nonnull
    public OutputStream getStdout() {
        return stdout;
    }

    public static class Builder extends BuilderSupport {

        private final String command;
        private final ImmutableMap.Builder<String, String> environment;
        private final OutputStream stderr;
        private final InputStream stdin;
        private final OutputStream stdout;

        public Builder(@Nonnull String command, @Nonnull InputStream stdin,
                       @Nonnull OutputStream stdout, @Nonnull OutputStream stderr) {
            this.command = requireNonBlank(command, "command");
            this.stderr = requireNonNull(stderr, "stderr");
            this.stdin = requireNonNull(stdin, "stdin");
            this.stdout = requireNonNull(stdout, "stdout");

            environment = ImmutableMap.builder();
        }

        @Nonnull
        public SshCommandContext build() {
            return new SshCommandContext(this);
        }

        @Nonnull
        public Builder environment(@Nonnull Map<String, String> values) {
            environment.putAll(requireNonNull(values, "environment"));

            return this;
        }

        @Nonnull
        public Builder environment(@Nonnull String key, @Nonnull String value) {
            environment.put(requireNonBlank(key, "key"), requireNonBlank(value, "value"));

            return this;
        }
    }
}
