package com.atlassian.bitbucket.shutdown;

import javax.annotation.Nonnull;
import java.time.Duration;

/**
 * Interface to be implemented by components who need to be informed about shutdown as soon as it is initiated. These
 * components will be given the opportunity to start shutdown, to allow them to shutdown
 * gracefully before the application terminates them. Components are allowed to complete shutdown for the maximum time
 * allowed per shutdown lifecycle phase which is defined at system level (configured with
 * {@code spring.lifecycle.timeout.per.shutdown.phase} system property). After that grace period is over, component will
 * be asked to shutdown immediately.
 *
 * @see ShutdownHookStatus
 * @since 7.8
 */
public interface ShutdownHook {

    /**
     * @return {@link ShutdownHookStatus status} of component to help decide if it is safe to terminate or not. This
     * method is polled repeatedly after {@link #startShutdown(Duration)} method is called. Once this method returns
     * {@link ShutdownHookStatus#IDLE IDLE} status, this component is assumed safe for shutdown.
     */
    @Nonnull
    ShutdownHookStatus getStatus();

    /**
     * Indicates that component should shutdown immediately as grace period for shutting down is over. Implementing this
     * method is optional in which case component can be forcefully shutdown in it's usual lifecycle phase or by
     * termination of JVM process.
     */
    default void shutdownNow() {
    }

    /**
     * Indicates that component should start shutdown process such that it stops performing new tasks, but leave currently
     * running tasks uninterrupted until asked to shutdown immediately. After starting shutdown process, it can be
     * waited for completion, otherwise, component will be repeatedly {@link #getStatus() polled} to check
     * {@link ShutdownHookStatus status}. This method will be called once per component during application shutdown,
     * provided {@code `server.shutdown`} system property is set as {@code 'graceful'}. This method will never be called
     * if {@code `server.shutdown`} is set as {@code 'immediate'}.
     *
     * @param timeout this is the time that can be taken to wait for shutdown to be complete after initiation. It is
     *                optional to use this timeout and just initiate the shutdown without waiting for termination, in
     *                which case, polling will be done using {@link #getStatus()} until {@link ShutdownHookStatus#IDLE}.
     */
    void startShutdown(@Nonnull Duration timeout);
}