package com.atlassian.bitbucket.scm.ssh;

import com.atlassian.bitbucket.ssh.command.SshCommandContext;
import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableMap;

import javax.annotation.Nonnull;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Map;

import static java.util.Objects.requireNonNull;

/**
 * @since 5.12
 */
public class SshScmRequestContext implements ExitCodeCallback {

    private final String command;
    private final Map<String, String> environment;
    private final ExitCodeCallback exitCodeCallback;
    private final OutputStream stderr;
    private final InputStream stdin;
    private final OutputStream stdout;

    private SshScmRequestContext(Builder builder) {
        command = builder.command;
        environment = builder.environment.build();
        exitCodeCallback = builder.exitCodeCallback;
        stderr = builder.stderr;
        stdin = builder.stdin;
        stdout = builder.stdout;
    }

    @Nonnull
    public String getCommand() {
        return command;
    }

    @Nonnull
    public Map<String, String> getEnvironment() {
        return environment;
    }

    @Nonnull
    public ExitCodeCallback getExitCodeCallback() {
        return exitCodeCallback;
    }

    @Nonnull
    public OutputStream getStderr() {
        return stderr;
    }

    @Nonnull
    public InputStream getStdin() {
        return stdin;
    }

    @Nonnull
    public OutputStream getStdout() {
        return stdout;
    }

    @Override
    public void onExit(int code) {
        exitCodeCallback.onExit(code);
    }

    public static class Builder extends BuilderSupport {

        private final String command;
        private final ImmutableMap.Builder<String, String> environment;

        private ExitCodeCallback exitCodeCallback;
        private OutputStream stderr;
        private InputStream stdin;
        private OutputStream stdout;

        public Builder(@Nonnull SshCommandContext context, @Nonnull ExitCodeCallback exitCodeCallback) {
            this(requireNonNull(context, "context").getCommand(), context.getStdin(),
                    context.getStdout(), context.getStderr(), exitCodeCallback);

            environment.putAll(context.getEnvironment());
        }

        public Builder(@Nonnull SshScmRequestContext context) {
            this(requireNonNull(context, "context").getCommand(), context.getStdin(), context.getStdout(),
                    context.getStderr(), context.getExitCodeCallback());

            environment.putAll(context.getEnvironment());
        }

        public Builder(@Nonnull String command, @Nonnull InputStream stdin, @Nonnull OutputStream stdout,
                       @Nonnull OutputStream stderr, @Nonnull ExitCodeCallback exitCodeCallback) {
            this.command = requireNonBlank(command, "command");
            this.exitCodeCallback = requireNonNull(exitCodeCallback, "exitCodeCallback");
            this.stderr = requireNonNull(stderr, "stderr");
            this.stdin = requireNonNull(stdin, "stdin");
            this.stdout = requireNonNull(stdout, "stdout");

            environment = ImmutableMap.builder();
        }

        @Nonnull
        public SshScmRequestContext build() {
            return new SshScmRequestContext(this);
        }

        @Nonnull
        public Builder environment(@Nonnull Map<String, String> values) {
            environment.putAll(requireNonNull(values, "environment"));

            return this;
        }

        @Nonnull
        public Builder environment(@Nonnull String key, @Nonnull String value) {
            environment.put(requireNonBlank(key, "key"), requireNonBlank(value, "value"));

            return this;
        }

        @Nonnull
        public Builder exitCodeCallback(@Nonnull ExitCodeCallback value) {
            exitCodeCallback = requireNonNull(value, "exitCodeCallback");

            return this;
        }

        @Nonnull
        public Builder stderr(@Nonnull OutputStream value) {
            stderr = requireNonNull(value, "stderr");

            return this;
        }

        @Nonnull
        public Builder stdin(@Nonnull InputStream value) {
            stdin = requireNonNull(value, "stdin");

            return this;
        }

        @Nonnull
        public Builder stdout(@Nonnull OutputStream value) {
            stdout = requireNonNull(value, "stdout");

            return this;
        }
    }
}
