package com.atlassian.bitbucket.scm.pull;

import com.atlassian.bitbucket.commit.CommitCallback;
import com.atlassian.bitbucket.content.ChangeCallback;
import com.atlassian.bitbucket.content.DiffContentCallback;
import com.atlassian.bitbucket.pull.PullRequest;
import com.atlassian.bitbucket.pull.PullRequestMergeOutcome;
import com.atlassian.bitbucket.pull.PullRequestMergeResult;
import com.atlassian.bitbucket.repository.Branch;
import com.atlassian.bitbucket.scm.Command;
import com.atlassian.bitbucket.scm.SimpleCommand;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;

import static java.util.Objects.requireNonNull;

public interface PluginPullRequestCommandFactory {

    /**
     * @param parameters describes the repository and provides a context for the command to use to retrieve
     *                   affected rescopes and apply updates
     * @return a command which, when called, will analyse any rescopes from the provided context
     * @since 4.5
     */
    @Nonnull
    Command<Void> bulkRescope(@Nonnull BulkRescopeCommandParameters parameters);

    /**
     * @param pullRequest the pull request to stream changes for
     * @param parameters  parameters describing the changes to stream
     * @param callback    a callback to receive changes
     * @return a command which, when called, will stream changes for the specified pull request
     * @since 6.0
     */
    @Nonnull
    Command<Void> changes(@Nonnull PullRequest pullRequest, @Nonnull PullRequestChangesCommandParameters parameters,
                          @Nonnull ChangeCallback callback);

    /**
     * @param pullRequest describes the refs being compared
     * @param parameters parameters providing context for the pull request commits
     * @param callback    a callback to receive commits which are unique to the {@link PullRequest#getFromRef from ref}
     * @return a command which, when called, will stream commits referenced by the {@link PullRequest#getFromRef from
     *         ref} which are not referenced by the {@link PullRequest#getToRef to ref}
     * @since 6.0
     */
    @Nonnull
    Command<Void> commits(@Nonnull PullRequest pullRequest, @Nonnull PullRequestCommitsCommandParameters parameters,
                          @Nonnull CommitCallback callback);

    /**
     * Deletes repository data associated with the specified pull request.
     * <p>
     * SCMs <i>must not</i> delete the {@link PullRequest#getFromRef from} or {@link PullRequest#getToRef to} refs.
     * This command is intended to delete any internal, SCM-specific data the SCM tracks for pull requests. For
     * example, if an SCM maintains additional commits or merges, or cached data such as change, diff or conflict
     * details used to speed up other operations, or extra refs used to allow that data to be fetched, those are the
     * types of data implementations of this command should remove.
     * <p>
     * Even if an SCM does not track any additional data beyond the refs involved in the pull request, the returned
     * {@link Command} must be non-{@code null}. Such SCMs may wish to leverage the default implementation, which
     * returns a command that does nothing when called.
     *
     * @param pullRequest the pull request to delete
     * @param parameters  parameters providing context for the deleted pull request
     * @return a command which, when called, will delete the specified pull request, releasing any artifacts,
     *         such as refs or objects, that are associated with it
     * @since 5.1
     */
    @Nonnull
    default Command<Void> delete(@Nonnull PullRequest pullRequest,
                                 @Nonnull PullRequestDeleteCommandParameters parameters) {
        return new SimpleCommand<Void>() {

            @Override
            public Void call() {
                return null;
            }
        };
    }

    @Nonnull
    Command<Void> diff(@Nonnull PullRequest pullRequest, @Nonnull PullRequestDiffCommandParameters parameters,
                       @Nonnull DiffContentCallback callback);

    @Nonnull
    Command<PullRequestEffectiveDiff> effectiveDiff(@Nonnull PullRequest pullRequest);

    @Nonnull
    Command<Branch> merge(@Nonnull PullRequest pullRequest, @Nonnull PullRequestMergeCommandParameters parameters);

    /**
     * Attempt to merge the pull request to determine what the expected result of actually merging it would be.
     * SCMs are not required to actually attempt the merge. Instead a {@link PullRequestMergeResult result} with
     * {@link PullRequestMergeOutcome#UNKNOWN UNKNOWN outcome} may be used to indicated that the merge
     * was not attempted or some unexpected error occurred while attempting the merge.
     *
     * @param pullRequest the pull request for which a dry run merge should be performed
     * @return a command which, when called, will return the result of attempting to merge the pull request
     * @since 5.0
     */
    @Nonnull
    Command<PullRequestMergeResult> tryMerge(@Nonnull PullRequest pullRequest);

    /**
     * Implementors: The default implementation of this method will be removed in 9.0. SCMs are <i>required</i> to
     * implement this command for themselves; they cannot rely on the default.
     *
     * @param parameters provides the pull requests for which the pull request refs should be brought into sync with
     *                   the current pull request state
     * @return a command which can be used to update the pull request refs synchronously or asynchronously
     * @since 7.12
     */
    @Nonnull
    default Command<Void> updateRefs(@Nonnull UpdatePullRequestRefsCommandParameters parameters) {
        Logger log = LoggerFactory.getLogger(getClass());
        return new SimpleCommand<Void>() {

            @Override
            public Void call() {
                String scmId = requireNonNull(parameters, "parameters").getPullRequests().stream()
                        .map(pullRequest -> pullRequest.getToRef().getRepository().getScmId())
                        .findFirst()
                        .orElse("unknown");
                log.debug("PluginPullRequestCommandFactory.updateRefs is not implemented for SCM {}", scmId);
                return null;
            }
        };
    }
}
