package com.atlassian.bitbucket.scm.pull;

import com.atlassian.bitbucket.hook.repository.RepositoryMergeCheck;
import com.atlassian.bitbucket.pull.PullRequest;
import com.atlassian.bitbucket.pull.PullRequestMergeRequest;
import com.atlassian.bitbucket.pull.PullRequestService;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Map;

/**
 * Represents a request to merge a pull request, intended to allow {@link RepositoryMergeCheck}s to validate the
 * intended merge and, potentially, {@link #veto} it.
 *
 * @see MergeRequestCheckService
 * @see PullRequestMergeRequest PullRequestMergeRequest
 */
public interface MergeRequest {

    /**
     * Retrieves any additional context that was supplied when the merge was requested. This context is intended to
     * allow plugin developers to implement their own behavior around pull request merges. As a result, its contents
     * are <i>unspecified</i>; they depend on the installed plugins.
     * <p>
     * On {@link #isDryRun dry runs}, the context will <i>always</i> be empty.
     *
     * @return additional context provided when the merge was requested, if any
     */
    @Nonnull
    Map<String, Object> getContext();

    /**
     * Retrieves the proposed commit message, if one was provided.
     * <p>
     * On {@link #isDryRun dry runs}, the commit message will <i>always</i> be {@code null}.
     *
     * @return the proposed commit message, which will be {@code null} for {@link PullRequestService#canMerge canMerge}
     *         checks and <i>may</i> be {@code null} for {@link PullRequestService#merge merge} requests
     */
    @Nullable
    String getMessage();

    /**
     * @return the pull request to be merged
     */
    @Nonnull
    PullRequest getPullRequest();

    /**
     * Retrieves a flag indicating whether this request is for a {@link PullRequestService#canMerge canMerge} check
     * or a real {@link PullRequestService#merge merge}.
     * <p>
     * This flag is intended to allow {@link RepositoryMergeCheck}s to apply different validation, if necessary, between
     * {@code canMerge} and {@code merge} requests. For example, a {@link RepositoryMergeCheck} to validate the
     * {@link #getMessage commit message} would not be applied on a dry run, because the commit message is only
     * supplied when the merge is actually performed.
     *
     * @return {@code true} if this is a {@link PullRequestService#canMerge canMerge} check; otherwise, {@code false}
     *         if it is a {@link PullRequestService#merge merge} request
     */
    boolean isDryRun();

    /**
     * @return {@code true} if a previous {@link RepositoryMergeCheck} has already {@link #veto vetoed} the merge;
     *         otherwise, {@code false}
     */
    boolean isVetoed();

    /**
     * If called, signifies that the calling check wishes to veto the merge.
     *
     * @param summaryMessage a simple summary of why the merge is being vetoed
     * @param detailedMessage a detailed explanation of why the merge is being vetoed
     */
    void veto(@Nonnull String summaryMessage, @Nonnull String detailedMessage);
}
