package com.atlassian.bitbucket.scm.pull;

import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.scm.AbstractCommandParameters;

import javax.annotation.Nonnull;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Objects.requireNonNull;

/**
 * @since 4.5
 */
public class BulkRescopeCommandParameters extends AbstractCommandParameters {

    private final Repository repository;
    private final BulkRescopeContext rescopeContext;

    private BulkRescopeCommandParameters(BulkRescopeContext rescopeContext, Repository repository) {
        this.repository = repository;
        this.rescopeContext = rescopeContext;
    }

    @Nonnull
    public Repository getRepository() {
        return repository;
    }

    @Nonnull
    public BulkRescopeContext getRescopeContext() {
        return rescopeContext;
    }

    public static class Builder {

        private Repository repository;
        private BulkRescopeContext rescopeContext;

        public Builder() {
        }

        public Builder(@Nonnull BulkRescopeCommandParameters parameters) {
            this();

            repository = parameters.getRepository();
            rescopeContext = parameters.getRescopeContext();
        }

        @Nonnull
        public BulkRescopeCommandParameters build() {
            checkState(repository != null, "The repository for which pull requests should be rescoped is required");
            checkState(rescopeContext != null, "The bulk rescope context is required");

            return new BulkRescopeCommandParameters(rescopeContext, repository);
        }

        @Nonnull
        public Builder repository(@Nonnull Repository value) {
            repository = requireNonNull(value, "repository");

            return this;
        }

        @Nonnull
        public Builder rescopeContext(@Nonnull BulkRescopeContext value) {
            rescopeContext = requireNonNull(value, "rescopeContext");

            return this;
        }
    }
}
