package com.atlassian.bitbucket.scm.integrity;

import com.atlassian.bitbucket.i18n.KeyedMessage;
import com.atlassian.bitbucket.repository.Repository;

import javax.annotation.Nonnull;

/**
 * Used by an SCM to report integrity check results as they become available.
 *
 * @since 4.12
 */
public interface IntegrityCheckCallback {

    /**
     * Called when the SCM encounters a problem that cannot be expressed in terms of pull requests or
     * repositories.
     *
     * @param message a detailed description that can be used by administrators to debug the problem
     */
    void onError(@Nonnull KeyedMessage message);

    /**
     * Called when the SCM encounters a problem while checking the integrity of a specific repository.
     *
     * @param repository the repository in which the error originated
     * @param message a detailed description that can be used by administrators to debug the problem
     */
    void onError(@Nonnull Repository repository, @Nonnull KeyedMessage message);

    /**
     * Invoked when the SCM encounters a pull request that does not exist according to the host application.
     *
     * @param repository the repository in which the pull request resides
     * @param pullRequestId the ID of the pull request
     */
    void onExtraPullRequest(@Nonnull Repository repository, long pullRequestId);

    /**
     * Invoked when the required SCM-level data for a pull request is missing.
     *
     * @param repository the repository in which the pull request resides
     * @param pullRequestId the ID of the pull request
     */
    void onMissingPullRequest(@Nonnull Repository repository, long pullRequestId);
}
