package com.atlassian.bitbucket.scm.http;

import com.atlassian.bitbucket.project.Project;
import com.atlassian.bitbucket.repository.Repository;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.StringUtils.*;

/**
 * Utility class for parsing project namespaces, project keys and repository slugs from scm urls - HTTP and SSH.
 *
 * <p>
 * This class only works with paths that begin with: <ul>
 *  <li>&lt;baseUrl&gt;/PROJECT_KEY/REPOSITORY_SLUG/</li>
 *  <li>&lt;baseUrl&gt;/PROJECT_NAMESPACE/PROJECT_KEY/REPOSITORY_SLUG/</li>
 *  <li>&lt;baseUrl&gt;/PROJECT_KEY/REPOSITORY_SLUG.git/ (the .git suffix is not captured)</li>
 *  <li>&lt;baseUrl&gt;/PROJECT_NAMESPACE/PROJECT_KEY/REPOSITORY_SLUG.git/ (the .git suffix is not captured)</li>
 * </ul>
 * <p>
 * This class captures any {@link #getPathSuffix() path suffix} that did not describe the repository but still may be
 * useful to identify.
 */
public class RepositoryUrlFragment {

    private static final String SHORT_HOSTING_URL = "/?(?<projKey>[^/]+)/(?<repoSlug>[^/]+?)(?:\\.git)?+(?<restOfPath>/.*|$)";
    private static final String NAMESPACED_HOSTING_URL = "/?(?<namespace>[^/]+)" + SHORT_HOSTING_URL;
    private static final Pattern PATTERN_SHORT_HOSTING_URL = Pattern.compile(SHORT_HOSTING_URL);
    private static final Pattern PATTERN_NAMESPACED_HOSTING_URL = Pattern.compile(NAMESPACED_HOSTING_URL);

    private final String pathSuffix;
    private final String projectNamespace;
    private final String projectKey;
    private final String repositorySlug;

    private RepositoryUrlFragment(String projectKey, String repositorySlug, String pathSuffix) {
        this.pathSuffix = trimToNull(pathSuffix);
        this.projectNamespace = null;
        this.projectKey = requireNonNull(projectKey, "projectKey");
        this.repositorySlug = requireNonNull(repositorySlug, "repositorySlug");
    }

    /**
     * Extracts the repository URL fragment from the given path info.
     *
     * @param pathInfo the path info as provided by the {@link HttpServletRequest#getPathInfo} method.
     * @return a new repository URL fragment, or null if the path info does not contain a repository URL fragment
     * @throws NullPointerException if the path info is null
     */
    @Nullable
    public static RepositoryUrlFragment fromPathInfo(@Nonnull String pathInfo) {
        Matcher m = PATTERN_SHORT_HOSTING_URL.matcher(requireNonNull(pathInfo, "pathInfo"));
        return m.find() ? new RepositoryUrlFragment(
                m.group("projKey"), m.group("repoSlug"), m.group("restOfPath")) : null;
    }

    /**
     * Extracts the namespaced repository URL fragment from the given path info.
     *
     * @param pathInfo the path info as provided by the {@link HttpServletRequest#getPathInfo} method.
     * @return a new repository URL fragment, or null if the path info does not contain a namespaced repository URL fragment
     * @throws NullPointerException if the path info is null
     *
     * @since 4.2
     * @deprecated in 7.18 for removal <em>without replacement</em> in 9.0
     */
    @Deprecated
    @Nullable
    public static RepositoryUrlFragment fromNamespacedPathInfo(@Nonnull String pathInfo) {
        Matcher m = PATTERN_NAMESPACED_HOSTING_URL.matcher(requireNonNull(pathInfo, "pathInfo"));
        return m.find() ? new RepositoryUrlFragment(
                m.group("projKey"), m.group("repoSlug"), m.group("restOfPath")) : null;
    }

    /**
     * @param repository the repository to create a fragment for
     * @return a URL fragment for the specified repository
     */
    @Nonnull
    public static RepositoryUrlFragment fromRepository(@Nonnull Repository repository) {
        Project project = requireNonNull(repository, "repository").getProject();
        String projectKey = project.getKey();

        return new RepositoryUrlFragment(projectKey, repository.getSlug(), null);
    }

    /**
     * @param projectKey     the project key to use
     * @param repositorySlug the repository slug to use
     * @return a URL fragment for the specified projectKey and repositorySlug
     *
     * @since 6.7
     */
    @Nonnull
    public static RepositoryUrlFragment fromKeyAndSlug(@Nonnull String projectKey, @Nonnull String repositorySlug) {
        requireNonNull(projectKey, "projectKey");
        requireNonNull(repositorySlug, "repositorySlug");

        return new RepositoryUrlFragment(projectKey, repositorySlug, null);
    }

    /**
     * @return the suffix after the namespace / project key / repository part of the path or null if there was none
     * @since 5.1
     */
    @Nullable
    public String getPathSuffix() {
        return pathSuffix;
    }

    @Nonnull
    public String getProjectKey() {
        return projectKey;
    }

    /**
     * @deprecated in 7.18 for removal <em>without replacement</em> in 9.0
     */
    @Deprecated
    @Nullable
    public String getProjectNamespace() {
        return projectNamespace;
    }

    @Nonnull
    public String getRepositorySlug() {
        return repositorySlug;
    }

    /**
     * Formats this fragment as a path suitable for use with a {@code URI}, such as with {@code URI.resolve(String)},
     * optionally pre-pending a provided base path. The {@link #getPathSuffix() path suffix} nor the .git
     * extension if present in the original path are output here.
     *
     * @param basePath the base path to prepend, or {@code null}
     * @return a URI-suitable path
     */
    @Nonnull
    public String toPath(@Nullable String basePath) {
        return toPath(basePath, false);
    }

    /**
     * Formats this fragment as a path suitable for use with a {@code URI}, such as with {@code URI.resolve(String)},
     * optionally pre-pending a provided base path. The {@link #getPathSuffix() path suffix} is included if
     * the {@code includeSuffix} parameter is {@code true}.
     *
     * @param basePath the base path to prepend, or {@code null}
     * @param includeSuffix whether to include the suffix
     * @return a URI-suitable path
     * @since 7.18
     */
    @Nonnull
    public String toPath(@Nullable String basePath, boolean includeSuffix) {
        StringBuilder builder = new StringBuilder();
        if (isEmpty(basePath)) {
            builder.append('/');
        } else {
            builder.append(basePath);
            if (!basePath.endsWith("/")) {
                builder.append('/');
            }
        }

        return builder.append(toPath(includeSuffix)).toString();
    }

    /**
     * @return {@link #getProjectKey() KEY}/{@link #getRepositorySlug() slug} for URLs on primary instances,
     *         {@link #getProjectNamespace()}() NAMESPACE}/{@link #getProjectKey() KEY}/{@link #getRepositorySlug() slug} on mirrors
     */
    @Nonnull
    @Override
    public String toString() {
        return toPath(null, false);
    }

    private String toPath(boolean includeSuffix) {
        StringBuilder builder = new StringBuilder();

        builder.append(projectKey)
                .append('/')
                .append(repositorySlug);

        if (includeSuffix && isNotEmpty(pathSuffix)) {
            builder.append('/').append(pathSuffix);
        }

        return builder.toString().toLowerCase(Locale.ROOT);
    }
}
