package com.atlassian.bitbucket.scm.http;

import com.atlassian.bitbucket.scm.ScmRequest;

import javax.servlet.http.HttpServletRequest;

/**
 * Handles an HTTP request from an SCM client (e.g. a git clone or git pull)
 *
 * @see HttpScmRequestHandler
 * @see HttpScmRequestHandlerModuleDescriptor
 */
public interface HttpScmRequest extends ScmRequest {

    /**
     * Realm to be used in any Authentication challenges.
     */
    String BITBUCKET_REALM = "Atlassian Bitbucket";

    /**
     * Allows requests to indicate whether they support using Servlet 3's {@link HttpServletRequest#startAsync
     * asynchronous execution}.
     * <p>
     * <i>In general</i>, synchronous and asynchronous execution behave similarly. The primary difference is that
     * asynchronous execution is moved to a background executor thread where synchronous execution takes place
     * directly on a servlet container (e.g. Tomcat) thread. As a result, most implementations will not need to
     * do anything specific to "support" asynchronous execution.
     * <p>
     * Asynchronous execution does <i>not</i> mean <i>non-blocking</i> execution. Asynchronous execution can still
     * perform blocking reads and writes during processing. However, in order for a request to use Servlet 3.1's
     * {@link javax.servlet.ReadListener} and {@link javax.servlet.WriteListener} to perform non-blocking handling,
     * it <i>must</i> be running asynchronously. (This is a restriction from the Servlet spec, not one imposed by
     * this SPI.)
     * <p>
     * <b>Warning</b>: Just because a request <i>supports</i> asynchronous execution <i>does not guarantee</i> it
     * will be invoked asynchronously. Implementors <i>must</i> check {@link HttpServletRequest#isAsyncStarted} to
     * determine whether the request is running asynchronously, and the implementation <i>must</i> still support
     * synchronous execution. This restriction stems from the pluggable nature of the filter chain. Apps are free
     * to register custom filters in the chain, applied to any URL (including the {@code /scm} hosting URL), and
     * they can mark their filters as not supporting async. When that happens, it prevents requests from executing
     * asynchronously.
     *
     * @return {@code true} if this request can be executed {@link HttpServletRequest#startAsync asynchronously};
     *          otherwise, {@code false} to require standard synchronous execution
     * @since 7.4
     */
    default boolean isAsyncSupported() {
        return false;
    }
}
