package com.atlassian.bitbucket.scm.event;

import com.atlassian.bitbucket.scm.Scm;

import javax.annotation.Nonnull;
import java.util.EventObject;

import static java.util.Objects.requireNonNull;

/**
 * Common base for all events raised by {@link Scm SCM} implementations.
 */
public abstract class ScmEvent extends EventObject {

    private final Scm scm;

    /**
     * Constructs a new {@code ScmEvent}, initialising the event's {@code source} and setting the {@link Scm} from which
     * the event is being raised.
     *
     * @param source the component within the SCM raising the event
     * @param scm    the SCM raising the event
     * @throws NullPointerException if {@code source} or {@code scm} is {@code null}
     */
    public ScmEvent(@Nonnull Object source, @Nonnull Scm scm) {
        //Base class does this check as well, but raises an exception that is inconsistent with requireNonNull
        super(requireNonNull(source, "source"));

        this.scm = requireNonNull(scm, "scm");
    }

    /**
     * Retrieves the {@link Scm} raising the event.
     * <p>
     * Note: This may not match the {@code getSource()} value, which could be a component within the SCM rather than
     * the {@link Scm} itself.
     *
     * @return the SCM
     */
    @Nonnull
    public Scm getScm() {
        return scm;
    }
}
