package com.atlassian.bitbucket.scm.bulk;

import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.scm.Command;

import javax.annotation.Nonnull;

/**
 * A factory for creating commands which operate on bulk content. These are essentially variants of
 * {@link com.atlassian.bitbucket.scm.PluginCommandFactory standard commands} which have been optimized
 * to work on large sets of data at once. The intention is to give SCM implementors an opportunity to
 * leverage SCM-specific capabilities to gather data more efficiently than making repeated calls to
 * standard commands.
 *
 * @since 4.2
 */
public interface PluginBulkContentCommandFactory {

    /**
     * Streams commits from one or more repositories, providing a set of matching repositories for each.
     * <p>
     * Implementors: For performance reasons, it is expected that commits to be streamed will be identified using their
     * full {@link com.atlassian.bitbucket.commit.Commit#getId IDs}. Implementations are <i>not</i> required to attempt
     * to match short hashes. Where possible, implementations should <i>not</i> scale with the number of repositories
     * from which commits are being requested. Not matching short IDs is intended to help facilitate such optimization.
     *
     * @param repository one of the repositories for which commits should be streamed
     * @param parameters parameters describing the commits to stream
     * @param callback   a callback to receive the requested commits
     * @return a command which, when {@link Command#call called}, will stream commits to the provided callback
     * @since 5.8
     */
    @Nonnull
    Command<Void> commits(@Nonnull Repository repository, @Nonnull BulkCommitsCommandParameters parameters,
                          @Nonnull BulkCommitCallback callback);

    /**
     * @param repository the repository to get contents from
     * @param parameters to specify where to get contents from
     * @param callback   to process resulting content
     * @return the command
     */
    @Nonnull
    Command<Void> contents(@Nonnull Repository repository, @Nonnull BulkContentCommandParameters parameters,
                           @Nonnull BulkContentCallback callback);

    /**
     * Stream commits from one or more repositories, in <i>topological order</i>. Topological order means no parent is
     * output before <i>all</i> of its descendants have been output. It does <i>not</i> require that descendants be
     * streamed in <i>date</i> order, but SCMs may optionally do so (so long as topological order is retained).
     *
     * @param repository one of the repositories for which commits should be streamed
     * @param parameters parameters describing the commits to include in the traversal
     * @param callback   a callback to receive the requested commits
     * @return a command which, when {@link Command#call called}, will stream commits, topologically ordered, to the
     *         provided callback
     * @since 5.11
     */
    @Nonnull
    Command<Void> traverseCommits(@Nonnull Repository repository,
                                  @Nonnull BulkTraverseCommitsCommandParameters parameters,
                                  @Nonnull BulkTraversalCallback callback);
}
