package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.user.ApplicationUser;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.net.URI;

import static java.util.Objects.requireNonNull;

/**
 * Describes a {@link #getRepository() repository} and {@link #getBaseUrl() base URL} to use when
 * {@link ScmUrlFormatter#format(ScmUrlRequest) formatting a URL}. A context {@link #getUser() user}
 * may also be provided.
 */
public class ScmUrlRequest {

    private final URI baseUrl;
    private final Repository repository;
    private final ApplicationUser user;

    private ScmUrlRequest(Builder builder) {
        this.baseUrl = builder.baseUrl;
        this.repository = builder.repository;
        this.user = builder.user;
    }

    @Nonnull
    public URI getBaseUrl() {
        return baseUrl;
    }

    @Nonnull
    public Repository getRepository() {
        return repository;
    }

    @Nullable
    public ApplicationUser getUser() {
        return user;
    }

    public boolean hasUser() {
        return user != null;
    }

    public static class Builder {

        private final URI baseUrl;
        private final Repository repository;

        private ApplicationUser user;

        public Builder(@Nonnull Repository repository, @Nonnull URI baseUrl) {
            this.baseUrl = requireNonNull(baseUrl, "baseUrl");
            this.repository = requireNonNull(repository, "repository");
        }

        @Nonnull
        public ScmUrlRequest build() {
            return new ScmUrlRequest(this);
        }

        @Nonnull
        public Builder user(@Nullable ApplicationUser value) {
            user = value;

            return this;
        }
    }
}
