package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.i18n.KeyedMessage;
import com.atlassian.bitbucket.scm.event.ScmStatusChangedEvent;
import com.atlassian.bitbucket.scm.pull.PluginPullRequestCommandFactory;

import javax.annotation.Nullable;

/**
 * Describes the current {@link Scm#getStatus() status} of an {@link Scm} plugin.
 * <p>
 * SCMs which can change their status at runtime are <i>encouraged</i> to raise
 * {@link ScmStatusChangedEvent events} when their
 * status changes.
 *
 * @see ScmService
 * @see ScmStatusChangedEvent ScmStatusChangedEvent
 */
public interface ScmStatus {

    /**
     * Retrieves a message describing the SCM's status.
     * <p>
     * When {@link #isAvailable()} returns {@code true}, implementations of this method <i>may</i> return {@code null}.
     * However, they are <i>encouraged</i> to return a message describing the SCM implementation instead. For example,
     * they might return a message indicating the version of the SCM being used.
     * <p>
     * When {@link #isAvailable()} returns {@code false}, implementations are this are <i>required</i> to return a
     * non-{@code null} message describing the issue or issues affecting the SCM. The message should be as detailed
     * and thorough as possible to aid administrators in resolving the issue.
     *
     * @return a message describing the SCM's current status, which may be {@code null} if the SCM is available
     */
    @Nullable
    KeyedMessage getMessage();

    /**
     * Retrieves a flag indicating whether the SCM is available to process requests.
     * <p>
     * When {@code true}, the {@link Scm} is expected to support its full, normal functionality. This means, at a
     * minimum, the {@link PluginCommandFactory command factory} and other {@link Scm} methods must work correctly.
     * If the plugin does not support {@link PluginCommandBuilderFactory command builders} or
     * {@link PluginPullRequestCommandFactory pull requests}, whether the SCM is available
     * or not is <i>not</i> expected to be relevant.
     * <p>
     * When {@code false}, {@link ScmService} will prevent the system, or other plugins, from accessing the SCM, and
     * the SCM's normal functionality is not expected to be usable. Note that plugins may directly access the SCM,
     * bypassing the {@link ScmService}, and, when so doing, may not perform their own status checks. As a result, SCM
     * implementations are encouraged to guard against such misuse internally, as the system cannot prevent it.
     *
     * @return {@code true} if the SCM is available and functioning normally; otherwise, {@code false} if
     */
    boolean isAvailable();
}
