package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.ResourceBusyException;
import com.atlassian.bitbucket.repository.Repository;

import javax.annotation.Nonnull;
import java.io.IOException;

/**
 * A read or write request targeting a Stash repository. Instances are pass through a series of validators managed by
 * the {@link ScmRequestCheckService} to verify preconditions (e.g. permissions), before they are
 * {@link #handleRequest() handled}.
 */
public interface ScmRequest {

    /**
     * @return the repository that is the target of this action.
     */
    @Nonnull
    Repository getRepository();

    /**
     * Performs whatever logic is necessary to handle the request. The exact operations performed are both SCM and
     * RPC-specific, and left entirely to the implementor. Implementations may throw {@code IOException} to indicate
     * failures related to reading input or writing output during processing.
     *
     * @throws IOException if reading or writing fails during processing
     * @throws ResourceBusyException when the server is too busy to execute the request
     */
    void handleRequest() throws IOException;

    /**
     * @return true if this is a write operation (e.g. a push to a repository) or false if it is a read-only operation
     * (e.g. a pull or clone from a repository)
     */
    boolean isWrite();

    /**
     * Sends the provided error details to the client, using SCM- and RPC-specific techniques as necessary.
     *
     * @param summary       a brief summary of the error
     * @param detailMessage a more detailed explanation of the failure
     * @throws IOException if the error details cannot be written to the underlying output mechanism
     */
    void sendError(@Nonnull String summary, @Nonnull String detailMessage) throws IOException;
}
