package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.user.ApplicationUser;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;

/**
 * An extension point for declaring an {@link Scm Scm's} support for a protocol and for generating protocol-specific
 * clone URLs for {@link Repository repositories} of that {@link Repository#getScmId() kind}.
 * <p>
 * This extension point is <i>not</i> used to implement such a protocol, merely to indicate that it is supported.
 * <p>
 * <b>Note</b>: Implementations of this interface are required to be thread-safe.
 */
@ThreadSafe
public interface ScmProtocol {

    /**
     * @return the base URL to use when formatting {@link #getCloneUrl(Repository, ApplicationUser) clone URLs}, which
     *         must not be {@code null}
     * @since 5.0
     */
    @Nonnull
    String getBaseUrl();

    /**
     * Retrieves the clone URL specific to this protocol for the supplied {@link Repository repository} and, if
     * appropriate, tailored to the {@link ApplicationUser supplied user}. Implementations must handle the case where
     * there is no such user active.
     *
     * @param repository the repository to calculate the clone URL against
     * @param user the possibly null user to tailor the clone URL for if this is appropriate to the protocol
     * @return the calculated clone URL or null if this cannot be calculated for any reason
     */
    @Nullable
    String getCloneUrl(@Nonnull Repository repository, @Nullable ApplicationUser user);

    /**
     * @return the name of the protocol e.g. "ssh" or "http". The name must be lowercase.
     */
    @Nonnull
    String getName();

    /**
     * Determines if this protocol supports the specified {@link Scm SCM}.
     *
     * @param scm the scm
     * @return true if this protocol is supported for the supplied {@link Scm} or false otherwise
     */
    default boolean supports(@Nonnull Scm scm) {
        return supports(scm.getId());
    }

    /**
     * Determines if this protocol supports the {@link Scm SCM} with specified {@link Scm#getId ID}.
     *
     * @param scmId the {@link Scm#getId ID} of the scm
     * @return true if this protocol is supported for the supplied SCM or false otherwise
     * @since 5.0
     */
    boolean supports(@Nonnull String scmId);
}
