package com.atlassian.bitbucket.scm;

import javax.annotation.Nonnull;
import java.util.Set;

/**
 * Describes an SPI for allowing SCMs to expose user-selectable merge strategies. In addition to exposing a set of
 * SCM-specific merge strategies, SCMs must also provide a {@link #getHelpTopic() help topic/URL} to document how
 * each strategy works.
 * <p>
 * This SPI is an <i>optional</i> part of the {@link Scm#getMergeStrategies() SCM contract}. SCMs which implement
 * merge strategies should {@link ScmFeature#MERGE_STRATEGIES enable the feature}, or their strategies will not be
 * used.
 *
 * @since 4.9
 */
public interface PluginMergeStrategies {

    /**
     * Retrieves a help topic, which is used to {@link com.atlassian.bitbucket.help.HelpPathService#getPageUrl
     * look up a URL} for additional documentation about the available merge strategies. While each strategy
     * has its own {@link PluginMergeStrategy#getDescriptionKey description}, that is intended to be a short
     * summary of how the strategy works. The help documentation is expected to provide a richer, more in-depth
     * explanation of what the available strategies do.
     * <p>
     * There are two approaches for implementing this method:
     * <ul>
     *     <li>Return a key, like {@code "myscm.mergestrategies.help"}.
     *     <ul>
     *         <li>The value can be defined by registering an
     *         <a href="https://developer.atlassian.com/server/framework/atlassian-sdk/adding-resources-to-your-project">"i18n"
     *         resource</a> in {@code atlassian-plugin.xml}.
     *         The XML looks like: {@code &lt;resource type="i18n" name="..." location="..."/&gt;}</li>
     *         <li>If the SCM is already registering an "i18n" resource, the key/value can simply be added to
     *         the existing resource; no need to define a separate one.</li>
     *     </ul>
     *     </li>
     *     <li>Return a fixed URL.
     *     <ul>
     *         <li>If the {@link com.atlassian.bitbucket.help.HelpPathService#getPageUrl(String) URL lookup}
     *         doesn't return a value, the system assumes the topic value is the URL to use.</li>
     *         <li>This approach doesn't require an "i18n" resource, for SCMs which don't have one.</li>
     *     </ul>
     *     </li>
     * </ul>
     *
     * @return a topic, used to look up a help URL, or the help URL to use
     */
    @Nonnull
    String getHelpTopic();

    /**
     * Retrieves the set of supported {@link PluginMergeStrategy merge strategies}. The returned set may be fixed,
     * or it may vary (for example, some merge strategies might only be available when a specific version or version
     * range of the SCM is installed).
     * <p>
     * The returned {@code Set} is required to contain <i>at least one</i> strategy, and implementors are <i>strongly
     * encouraged</i> to support at least two. Merge strategies are not actually "selectable" when there is only one
     * option.
     *
     * @return a set containing <i>one or more</i> available strategies
     */
    @Nonnull
    Set<PluginMergeStrategy> getStrategies();
}
