package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.commit.Commit;
import com.atlassian.bitbucket.commit.LastModifiedCallback;
import com.atlassian.bitbucket.content.ArchiveFormat;
import com.atlassian.bitbucket.io.TypeAwareOutputSupplier;
import com.atlassian.bitbucket.repository.Branch;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.scm.signed.SignedObjectCallback;
import com.atlassian.bitbucket.scm.signed.SignedObjectsParameters;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * Provides backing functionality for the {@link ScmExtendedCommandFactory}.
 * <p>
 * This interface is intended to be <i>implemented</i> by plugin developers. Functionality provided by this interface
 * is intended to be consumed using the {@link ScmExtendedCommandFactory} exposed by the {@link ScmService}. Only the
 * system should ever deal with this interface directly.
 * <p>
 * This interface describes <i>optional</i> commands which an SCM implementor may provide to extend the functionality
 * of their SCM. Some features, like auto-merging between release branches and creating forks, rely on these optional
 * commands. If the SCM does not implement them, the features which depend on them will be disabled.
 *
 * @since 4.6
 */
public interface PluginExtendedCommandFactory {

    /**
     * Streams an archive of the repository at the specified commit, in the requested format. If paths are supplied,
     * the streamed archive <i>should</i> be filtered to only include files from the specified paths.
     * <p>
     * Implementors: This operation is <i>optional</i>. Implementations which do not support streaming archives should
     * return {@code null} and omit the {@link ScmFeature#ARCHIVE ARCHIVE} feature. For simplicity, this factory
     * provides a default implementation which returns {@code null}. SCMs which implement this command are
     * <i>required</i> to support streaming archives in <i>all</i> of the {@link ArchiveFormat standard formats}.
     * If the underlying SCM does not innately support one or more of the standard formats, the implementation is
     * <i>required</i> to augment the SCM and support them manually. If such augmentation is not possible, the SCM
     * <i>must not</i> implement this command.
     *
     * @param repository     the repository to stream the archive from
     * @param parameters     parameters describing the commit to archive and the format to produce the archive in
     * @param outputSupplier a supplier which, when given the archive content type, will provide an output stream
     * @return a command to stream an archive of the repository, or {@code null} if archiving is not supported
     * @see ScmFeature#ARCHIVE
     * @since 5.1
     */
    @Nullable
    default Command<Void> archive(@Nonnull Repository repository, @Nonnull ArchiveCommandParameters parameters,
                                  @Nonnull TypeAwareOutputSupplier outputSupplier) {
        return null;
    }

    /**
     * Adds a new file or updates the content of an existing file as described in the {@code parameters}.
     * <p>
     * Implementors: This feature is <i>optional</i>. Implementations which do not support updating the content of a file
     * should return {@code null} and  omit the {@link ScmFeature#EDIT_FILE EDIT_FILE} feature.
     * For simplicity, this factory provides a default implementation which returns {@code null}.
     *
     * @param repository the repository
     * @param parameters parameters describing the file path, file content, branch and the previous contentId of the file.
     * @return a command to edit a file, or {@code null} if editing file is not supported
     * @see ScmFeature#EDIT_FILE
     * @since 4.13
     */
    @Nullable
    default Command<Commit> editFile(@Nonnull Repository repository, @Nonnull EditFileCommandParameters parameters) {
        return null;
    }

    /**
     * Creates a fork of {@code repository}, stored in the {@link ForkCommandParameters#getFork specified fork}.
     * Similar to {@link PluginCommandFactory#create creating} a repository, the specified fork <i>must not</i>
     * exist on disk when this command is called. It is expected that the SCM will create it with contents based
     * on {@code repository}, which becomes the fork's {@link Repository#getOrigin origin}.
     * <p>
     * Implementors: This operation is <i>optional</i>. Implementations which do not support forking should return
     * {@code null} and omit the {@link ScmFeature#FORK FORK} feature. For simplicity, this factory provides a
     * default implementation which returns {@code null}.
     *
     * @param repository the fork repository
     * @param parameters parameters describing the origin repository to be forked
     * @return a command to create a fork of a repository, or {@code null} if forking is not supported
     * @see ScmFeature#FORK
     */
    @Nullable
    default Command<Void> fork(@Nonnull Repository repository, @Nonnull ForkCommandParameters parameters) {
        return null;
    }

    /**
     * Streams the latest commit to modify each file in the specified {@link LastModifiedCommandParameters#getPath()
     * path}, using the provided {@link LastModifiedCommandParameters#getCommitId() commit} as the starting point
     * for the traversal.
     * <p>
     * Implementors: This feature is <i>optional</i>. Implementations which do not support updating the default branch
     * should return {@code null} and  omit the {@link ScmFeature#LAST_MODIFIED LAST_MODIFIED} feature. For simplicity,
     * this factory provides a default implementation which returns {@code null}.
     *
     * @param repository the repository containing the starting commit
     * @param parameters parameters describing the path to stream modifications for, and the starting commit for
     *                   the traversal
     * @param callback   a callback to receive the latest commit for files in the specified path
     * @return a command to stream the latest commit to modify each files in the specified path, or {@code null} if
     *         streaming last modifications is not supported
     * @see ScmFeature#LAST_MODIFIED
     */
    @Nullable
    default Command<Void> lastModified(@Nonnull Repository repository,
                                       @Nonnull LastModifiedCommandParameters parameters,
                                       @Nonnull LastModifiedCallback callback) {
        return null;
    }

    /**
     * Merges the specified {@code {@link MergeCommandParameters#getFromCommitId() fromCommitId}} into the specified
     * {@code {@link MergeCommandParameters#getToBranch() toBranch}}.
     * <p>
     * SCMs are required to support the following values for {@code fromCommitId}:
     * <ul>
     *     <li>A 40-byte hash (Note: Short hashes are <i>not supported</i>)</li>
     *     <li>A fully-qualified branch name</li>
     *     <li>A short branch name</li>
     * </ul>
     * {@code toBranch}, as the parameter name suggests, <i>must</i> be a branch name, either fully qualified or short,
     * and the SCM should validate that that is the case; using a hash (full or short) is not supported.
     * <p>
     * Implementors: This feature is <i>optional</i>. Implementations which do not support merging should return
     * {@code null} and omit the {@link ScmFeature#MERGE MERGE} feature. For simplicity, this factory provides a
     * default implementation which returns {@code null}. Additionally, SCMs which support merging are not required to
     * support {@link MergeCommandParameters#getFromRepository() different source repositories} unless they offer
     * {@link ScmFeature#CROSS_REPOSITORY cross-repository} support. If a different from source repository is provided
     * and cross-repository operations are not supported, implementations
     * should throw an {@code UnsupportedOperationException}.
     *
     * @param repository the repository containing <i>both</i> the source commit and the target branch
     * @param parameters parameters describing the branches to merge
     * @return a command which, when executed, will merge the specified branch into the target branch, or {@code null}
     *         if merging is not supported
     * @throws UnsupportedOperationException if a {@link MergeCommandParameters#getFromRepository() source repository}
     *                                       is specified and {@link ScmFeature#CROSS_REPOSITORY cross-repository}
     *                                       operations are not supported
     * @see ScmFeature#CROSS_REPOSITORY
     * @see ScmFeature#MERGE
     */
    @Nullable
    default Command<Branch> merge(@Nonnull Repository repository, @Nonnull MergeCommandParameters parameters) {
        return null;
    }

    /**
     * Streams a patch of the specified repository for a given commit or commit range.
     * <p>
     * Implementors: This operation is <i>optional</i>. Implementations which do not support streaming patches should
     * return {@code null} and omit the {@link ScmFeature#PATCH PATCH} feature. For simplicity, this factory provides
     * a default implementation which returns {@code null}.
     *
     * @param repository     the repository from which to stream the patch
     * @param parameters     parameters describing the commit for which to stream the patch
     * @param outputSupplier a supplier which will provide an output stream
     * @return a command to stream a patch of the repository, or {@code null} if not supported
     * @see ScmFeature#PATCH
     * @since 6.7
     */
    @Nullable
    default Command<Void> patch(@Nonnull Repository repository, @Nonnull PatchCommandParameters parameters,
                                @Nonnull TypeAwareOutputSupplier outputSupplier) {
        return null;
    }

    /**
     * Pushes the repository to the {@link PushCommandParameters#getRemoteUrl() URL specified in the parameters}.
     *
     * @param repository the repository to push
     * @param parameters parameters describing the target of the push, optional authentication and which refs should
     *                   be included
     * @param callback   a callback that will receive ref updates, ref failures and progress updates
     * @return a command which, when executed, will perform a push of the repository to the URL specified, or
     *         {@code null} if not supported
     * @see ScmFeature#PUSH
     * @since 7.11
     */
    @Nullable
    default Command<Void> push(@Nonnull Repository repository, @Nonnull PushCommandParameters parameters,
                               @Nonnull PushCallback callback) {
        return null;
    }

    /**
     * Retrieves requested objects and if the object is signed, provides the signature and signed content to the
     * callback.
     *
     * @param repository the repository containing the requested objects
     * @param parameters describes which objects to retrieve
     * @param callback callback to receive the requested objects
     * @return a command which, when executed, will retrieve the requested objects and provide them to the provided
     *         callback
     * @throws FeatureUnsupportedScmException if the underlying SCM does not support retrieving signing objects
     * @see ScmFeature#SIGNED_OBJECTS
     *
     * @since 5.1
     */
    @Nullable
    default Command<Void> signedObjects(@Nonnull Repository repository, @Nonnull SignedObjectsParameters parameters,
                                        @Nonnull SignedObjectCallback callback) {
        return null;
    }

    /**
     * Sets the default branch for the repository to the {@link UpdateDefaultBranchCommandParameters#getBranchId()
     * named branch}.
     * <p>
     * Implementors: This feature is <i>optional</i>. Implementations which do not support updating the default branch
     * should return {@code null} and  omit the {@link ScmFeature#UPDATE_DEFAULT_BRANCH UPDATE_DEFAULT_BRANCH} feature.
     * For simplicity, this factory provides a default implementation which returns {@code null}.
     *
     * @param repository the repository whose default branch should be updated
     * @param parameters parameters describing the new branch to set as the default
     * @return a command to update the default branch, or {@code null} if updating the default branch is not supported
     * @see ScmFeature#UPDATE_DEFAULT_BRANCH
     */
    @Nullable
    default Command<Void> updateDefaultBranch(@Nonnull Repository repository,
                                              @Nonnull UpdateDefaultBranchCommandParameters parameters) {
        return null;
    }
}
