package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.scm.pull.ScmPullRequestCommandFactory;

import javax.annotation.Nonnull;

/**
 * When implemented by an {@link Scm SCM}, allows the {@link ScmService} to provide an {@link ScmCommandBuilder} that
 * can be used to build custom commands for advanced functionality. SCMs which do not support a command builder should
 * not implement this interface and should return {@code null} from {@link Scm#getCommandBuilderFactory()}.
 * <p>
 * SCMs implementing this interface are <i>required</i> to provide a free-form {@link ScmCommandBuilder}. They are also
 * <i>encouraged</i> to provide a type-safe builder API on top of that, to simplify creating custom commands correctly
 * and safely. For plugins which provide a type-safe builder, it is encouraged for them to use a sub-interface which
 * extends this one. For example:
 * <pre><code>
 *     //A type-safe builder for MyScm's diff command
 *     public interface DiffCommandBuilder extends CommandBuilderSupport&lt;DiffCommandBuilder&gt; {
 *         //A fluent API for constructing a diff command, exposing builder-style methods for setting
 *         //only parameters that are supported as valid arguments on the SCM's diff command
 *     }
 *
 *     //Adds methods for retrieving type-safe builders, in addition to the normal free-form builder methods
 *     public interface MyScmCommandBuilder extends ScmCommandBuilder&lt;MyScmCommandBuilder&gt; {
 *         //A type-safe builder for performing diffs
 *         DiffCommandBuilder diff();
 *
 *         //Other custom builders for other commands supported by the SCM
 *     }
 *
 *     //Overrides the builder(Repository) method using a covariant return type to return a builder supporting both
 *     //type-safe and free-form commands, instead of {@link ScmCommandBuilder}
 *     public interface MyScmCommandBuilderFactory extends PluginCommandBuilderFactory {
 *
 *         &#064;Nonnull
 *         &#064;Override
 *         MyScmCommandBuilder builder(&#064;Nullable Repository repository);
 *     }
 * </code></pre>
 * Notice that the type-safe builder extends from {@link CommandBuilderSupport}, <i>not</i> {@link CommandBuilder}. It
 * is encouraged that plugins which expose a type-safe builder API do not mix their type-safe API with their free-form
 * API, as doing so effectively defeats the purpose of providing a type-safe API and will make it more difficult for the
 * implementor to pass correct arguments. When a consumer is using the {@code MyScmCommandBuilder}, they may choose to
 * construct a diff command entirely using free-form arguments, or they may create a {@code DiffCommandBuilder} and
 * leverage its API to produce one with known-correct arguments.
 * <p>
 * The SCM plugin would then expose the class implementing their custom {@code MyScmCommandBuilderFactory} using a
 * {@code component} directive:
 * <pre><code>
 *     &lt;!-- Note: This should be in the SCM <u>provider's</u> atlassian-plugin.xml --&gt;
 *     &lt;component key="myScmCommandBuilderFactory"
 *                   class="com.example.DefaultMyScmCommandBuilderFactory"
 *                   public="true"&gt;
 *         &lt;interface&gt;com.example.MyScmCommandBuilderFactory&lt;/interface&gt;
 *     &lt;/component&gt;
 * </code></pre>
 * The {@code public="true"} allows other plugins to import the component. This approach allows other plugin developers
 * to import the SCM plugin's type-safe builder API with a {@code component-import} directive:
 * <pre><code>
 *     &lt;!-- Note: This should be in the SCM <u>consumer's</u> atlassian-plugin.xml --&gt;
 *     &lt;component-import key="myScmCommandBuilderFactory"
 *                          interface="com.example.MyScmCommandBuilderFactory"/&gt;
 * </code></pre>
 * <p>
 * Note: The system never uses the {@link ScmCommandBuilder} to perform SCM operations. SCM plugins are encouraged to
 * provide a command builder to allow other plugins to leverage SCM features not exposed by {@link ScmCommandFactory}
 * and {@link ScmPullRequestCommandFactory ScmPullRequestCommandFactory}
 */
public interface PluginCommandBuilderFactory {

    /**
     * Creates a {@link ScmCommandBuilder} which can be used to construct free-form commands to interact directly with
     * the underlying SCM implementation and perform custom functions on repositories.
     * <p>
     * Command builders are provided as a very powerful extension point for plugin developers, allowing them to create
     * new functionality that uses commands the application does not, or uses them in different ways. However, plugin
     * developers should be aware that:
     * <ul>
     *     <li>The system may support multiple versions of an SCM, and those versions may support different commands
     *     and/or arguments, and even the "same" command may produce different output between versions</li>
     *     <li>In a free-form builder, the command and arguments are not checked until the command is run; the system
     *     does not, and cannot, validate that the arguments are correct in advance</li>
     *     <li>Above all, <i>the system cannot verify the safety of commands being executed</i>; plugin developers
     *     <i>must</i> apply due diligence to ensure the operations they perform do not damage the repository</li>
     * </ul>
     * <p>
     * If a {@link Repository} is provided, the created {@link ScmCommandBuilder} will use the repository's directory
     * as its initial working directory. However, for commands that create repositories, such as {@code git init}, the
     * provided repository may be {@code null}. In such cases, the builder will use the default working directory. Note
     * that, after the builder has been created, it is possible to change the working directory on the instance, before
     * commands are constructed.
     * <p>
     * At a minimum, all {@code ScmCommandBuilderFactory} implementations are <i>required</i> to provide a free-form
     * builder. They are <i>encouraged</i> to provide a type-safe builder as well, by using a covariant return on their
     * implementation of this method to return a derived builder supporting both free-form and type-safe techniques.
     * How to do this is described in the class-level documentation.
     * <p>
     * Note: Per the contract of the {@link ScmCommandBuilder} API, the returned builder is <i>not thread-safe</i>. If
     * multiple threads require use of a builder, each thread should <i>always</i> create its own builder.
     *
     * @param repository the repository whose directory should be the initial working directory, or {@code null}, if no
     *                   repository exists, to use the default working directory
     * @return a new command builder
     */
    @Nonnull
    ScmCommandBuilder<?> builder(@Nonnull Repository repository);
}
