package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.repository.RepositoryForkRequest;
import com.atlassian.bitbucket.repository.RepositoryService;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.StringUtils.trimToNull;

/**
 * Describes the fork being created.
 * <p>
 * {@link RepositoryService#fork(RepositoryForkRequest)} is used to create forks. That service creates the fork
 * {@link Repository} in the database, and then calls {@link PluginExtendedCommandFactory#fork} to create it on
 * disk. As with other commands, the {@link Repository#getOrigin() origin} repository, which already exists both
 * in the database and on disk, is provided as the first argument when creating the fork {@link Command command}.
 * The <i>{@link #getFork fork}</i> repository, which only exists in the database, is provided in these parameters.
 *
 * @since 4.6
 */
public class ForkCommandParameters extends AbstractCommandParameters {

    private final String defaultBranch;
    private final Repository fork;

    private ForkCommandParameters(Builder builder) {
        defaultBranch = builder.defaultBranch;
        fork = builder.fork;
    }

    /**
     * Retrieves the <i>fork</i> repository, which is expected to have already been created in the database and
     * have a valid {@link Repository#getId() ID}. The SCM will use this {@link Repository} to create the fork
     * on disk and initialize it from its {@link Repository#getOrigin() origin}, which is supplied as the first
     * parameter to {@link PluginExtendedCommandFactory#fork} when creating the {@link Command}.
     *
     * @return the <i>fork</i> repository, which exists in the database but not on disk
     */
    @Nonnull
    public Repository getFork() {
        return fork;
    }

    /**
     * Retrieves the default branch that should be set for the fork after it's created. If the default branch
     * isn't specified, the fork will inherit its origin's default branch.
     *
     * @return the default branch to set, or {@code null} to inherit the origin's default branch
     * @since 7.5
     */
    @Nullable
    public String getDefaultBranch() {
        return defaultBranch;
    }

    public static class Builder {

        private final Repository fork;

        private String defaultBranch;

        public Builder(@Nonnull Repository fork) {
            this.fork = requireNonNull(fork, "fork");
        }

        @Nonnull
        public ForkCommandParameters build() {
            return new ForkCommandParameters(this);
        }

        /**
         * Specifies the default branch that should be set after the fork is created. If no default branch is
         * specified, the fork will inherit its origin's default branch.
         *
         * @param value the default branch name, or {@code null} to inherit the origin's default branch
         * @return {@code this}
         * @since 7.5
         */
        @Nonnull
        public Builder defaultBranch(@Nullable String value) {
            defaultBranch = trimToNull(value);

            return this;
        }
    }
}
