package com.atlassian.bitbucket.migration;

import com.atlassian.bitbucket.scope.*;
import com.google.common.base.MoreObjects;

import javax.annotation.Nonnull;
import java.util.Arrays;
import java.util.Objects;
import java.util.Optional;

public class StandardMigrationEntityType<K> implements MigrationEntityType<K> {

    public static final MigrationEntityType<Long> ATTACHMENT = new StandardMigrationEntityType<>("attachment");
    public static final MigrationEntityType<Long> COMMENT = new StandardMigrationEntityType<>("comment");
    public static final MigrationEntityType<Integer> PROJECT = new StandardMigrationEntityType<>("project");
    public static final MigrationEntityType<Integer> REPOSITORY = new StandardMigrationEntityType<>("repository");
    public static final MigrationEntityType<Integer> USER = new StandardMigrationEntityType<>("user");

    private final String name;

    private StandardMigrationEntityType(@Nonnull String name) {
        this.name = name;
    }

    @Nonnull
    public static Optional<MigrationEntityType<?>> ofName(@Nonnull String name) {
        return Arrays.stream(values())
                .filter(type -> Objects.equals(type.getName(), name))
                .findAny();
    }

    @Nonnull
    public static Optional<MigrationEntityType<Integer>> ofScope(@Nonnull Scope scope) {
        return scope.accept(new ScopeVisitor<Optional<MigrationEntityType<Integer>>>() {
            @Override
            public Optional<MigrationEntityType<Integer>> visit(@Nonnull RepositoryScope scope) {
                return Optional.of(REPOSITORY);
            }

            @Override
            public Optional<MigrationEntityType<Integer>> visit(@Nonnull ProjectScope scope) {
                return Optional.of(PROJECT);
            }

            @Override
            public Optional<MigrationEntityType<Integer>> visit(@Nonnull GlobalScope scope) {
                return Optional.empty();
            }
        });
    }

    @Nonnull
    public static MigrationEntityType<?>[] values() {
        return new MigrationEntityType[]{
                PROJECT, REPOSITORY, USER
        };
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        StandardMigrationEntityType<?> that = (StandardMigrationEntityType<?>) o;
        return Objects.equals(getName(), that.getName());
    }

    @Nonnull
    @Override
    public String getName() {
        return name;
    }

    @Override
    public int hashCode() {
        return Objects.hash(getName());
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("name", name)
                .toString();
    }
}
