package com.atlassian.bitbucket.migration;

import com.atlassian.bitbucket.io.IoConsumer;

import javax.annotation.Nonnull;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;

/**
 * @since 5.13
 */
public interface SequentialArchive extends Closeable {

    /**
     * Add an entry to an archive.
     *
     * @param logicalPathInBundle where in the archive this entry should be written
     * @param outputStream        used to write the contents of the entry
     * @throws IOException        if an error was encountered while performing IO
     * @since 5.13
     */
    void addEntry(@Nonnull Path logicalPathInBundle, @Nonnull IoConsumer<OutputStream> outputStream) throws IOException;

    /**
     * Add an entry to an archive.
     *
     * @param destinationInBundle where in the archive this entry should be written
     * @param sourceOnDisk        source for entry data
     * @param filter              filter files from being written to archive
     * @param attempts            the number of times to retry this operation in the case of failure
     * @throws IOException        if an error was encountered while performing IO
     * @since 5.13
     */
    void addPathFromDisk(@Nonnull Path destinationInBundle, @Nonnull Path sourceOnDisk,
                         @Nonnull PathMatcher filter, int attempts) throws IOException;

    /**
     * Add an entry to an archive.
     *
     * @param destinationInBundle where in the archive this entry should be written
     * @param sourceOnDisk        source for entry data
     * @param filter              filter files from being written to archive
     * @throws IOException        if an error was encountered while performing IO
     * @since 5.14
     */
    default void addPathFromDisk(@Nonnull Path destinationInBundle, @Nonnull Path sourceOnDisk,
                                 @Nonnull PathMatcher filter) throws IOException {
        addPathFromDisk(destinationInBundle, sourceOnDisk, filter, 1);
    }

    /**
     * Add an entry to an archive.
     *
     * @param destinationInBundle where in the archive this entry should be written
     * @param sourceOnDisk        source for entry data
     * @throws IOException        if an error was encountered while performing IO
     * @since 5.14
     */
    default void addPathFromDisk(@Nonnull Path destinationInBundle, @Nonnull Path sourceOnDisk) throws IOException {
        addPathFromDisk(destinationInBundle, sourceOnDisk, path -> true, 1);
    }

    /**
     * Add an entry to an archive.
     *
     * @param destinationInBundle where in the archive this entry should be written
     * @param sourceOnDisk        source for entry data
     * @param attempts            the number of times to retry this operation in the case of failure
     * @throws IOException        if an error was encountered while performing IO
     * @since 5.14
     */
    default void addPathFromDisk(@Nonnull Path destinationInBundle, @Nonnull Path sourceOnDisk, int attempts)
            throws IOException {
        addPathFromDisk(destinationInBundle, sourceOnDisk, path -> true, attempts);
    }

    /**
     * Add an entry to an archive.
     *
     * @param sourceOnDisk source for entry data
     * @throws IOException if an error was encountered while performing IO
     * @since 5.13
     */
    default void addPathFromDisk(@Nonnull Path sourceOnDisk) throws IOException {
        addPathFromDisk(sourceOnDisk, path -> true);
    }

    /**
     * Add an entry to an archive.
     *
     * @param sourceOnDisk source for entry data
     * @param attempts     the number of times to retry this operation in the case of failure
     * @throws IOException if an error was encountered while performing IO
     * @since 5.14
     */
    default void addPathFromDisk(@Nonnull Path sourceOnDisk, int attempts) throws IOException {
        addPathFromDisk(sourceOnDisk, path -> true, attempts);
    }

    /**
     * Add an entry to an archive.
     *
     * @param sourceOnDisk source for entry data
     * @param filter       filter files from being written to archive
     * @param attempts     the number of times to retry this operation in the case of failure
     * @throws IOException if an error was encountered while performing IO
     * @since 5.14
     */
    default void addPathFromDisk(@Nonnull Path sourceOnDisk, @Nonnull PathMatcher filter,
                                 int attempts) throws IOException {
        addPathFromDisk(Paths.get(""), sourceOnDisk, filter, attempts);
    }

    /**
     * Add an entry to an archive.
     *
     * @param sourceOnDisk source for entry data
     * @param filter       filter files from being written to archive
     * @throws IOException if an error was encountered while performing IO
     * @since 5.13
     */
    default void addPathFromDisk(@Nonnull Path sourceOnDisk, @Nonnull PathMatcher filter) throws IOException {
        addPathFromDisk(Paths.get(""), sourceOnDisk, filter, 1);
    }
}
