package com.atlassian.bitbucket.migration;

import com.atlassian.bitbucket.scm.BaseWeightedModuleDescriptor;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginParseException;
import com.atlassian.plugin.StateAware;
import com.atlassian.plugin.descriptors.AbstractModuleDescriptor;
import com.atlassian.plugin.module.ModuleFactory;
import com.atlassian.plugin.util.validation.ValidationPattern;
import org.dom4j.Element;

import javax.annotation.Nonnull;
import java.util.Optional;

import static com.atlassian.plugin.util.validation.ValidationPattern.test;
import static java.util.Optional.ofNullable;

/**
 * Module descriptor for providing {@link Exporter} and {@link Importer} implementations.
 * <p>
 * Usage:
 * <pre><code>
 *     &lt;migration-handler key="customExporter" weight="120"&gt;
 *         &lt;exporter class="com.example.bitbucket.internal.migration.DefaultCoreExporter" /&gt;
 *         &lt;importer class="com.example.bitbucket.internal.migration.DefaultCoreImporter" /&gt;
 *     &lt;/migration&gt;
 * </code></pre>
 *
 * <p>
 * The (optional) {@code weight} attribute defines the order in which {@link Exporter} and {@link Importer}
 * implementations will be called. Lower weights are called earlier and the default weight is 100.
 * Third party plugins <b>must</b> use a weight of 100 or higher
 * </p>
 * The {@code key} attribute on {@code <migration>} and the {@code class} attribute on {@code <exporter>}
 * and {@code <importer>} are <b>required</b>.
 *
 * @see Exporter
 * @see Importer
 * @since 5.13
 */
public class MigrationHandlerModuleDescriptor extends BaseWeightedModuleDescriptor<Void> {

    public static final String XML_ELEMENT_NAME = "migration-handler";

    private final NestedModuleDescriptor<Exporter> exporter;

    private volatile NestedModuleDescriptor<Importer> importer;

    public MigrationHandlerModuleDescriptor(ModuleFactory moduleFactory) {
        super(moduleFactory, 100);

        exporter = new NestedModuleDescriptor<>(moduleFactory);
        importer = new NestedModuleDescriptor<>(moduleFactory);
    }

    @Override
    public void disabled() {
        exporter.disabled();
        if (importer != null) {
            importer.disabled();
        }
    }

    @Nonnull
    public Exporter getExporter() {
        return exporter.getModule();
    }

    @Nonnull
    public Optional<Importer> getImporter() {
        return ofNullable(importer).map(NestedModuleDescriptor::getModule);
    }

    @Override
    public Void getModule() {
        return null;
    }

    @Override
    public void init(@Nonnull Plugin plugin, @Nonnull Element element) throws PluginParseException {
        super.init(plugin, element);

        canOnlyHaveOneChildElementOfName(element, "exporter");
        canOnlyHaveOneChildElementOfName(element, "importer");
        Element exporterElement = element.element("exporter");
        if (exporterElement == null) {
            throw new PluginParseException(
                    "Element <" + XML_ELEMENT_NAME + "> needs to define an <exporter> child element");
        }
        exporter.init(plugin, exporterElement);

        Element importerElement = element.element("importer");
        if (importerElement == null) {
            importer = null;
        } else {
            importer.init(plugin, importerElement);
        }
    }

    private void canOnlyHaveOneChildElementOfName(@Nonnull Element element, @Nonnull String childElementName) {
        if (element.elements(childElementName).size() > 1) {
            throw new PluginParseException("Element <" + XML_ELEMENT_NAME + "> " +
                    "can only have one <" + childElementName + "> child element");
        }
    }

    /**
     * Nested module descriptor type to allow constructing Exporter and Importer instances using the standard
     * ModuleFactory mechanism. This ensures that class="bean:beanName" style definitions are also supported
     *
     * @param <T> the module type
     */
    private static class NestedModuleDescriptor<T> extends AbstractModuleDescriptor<T> {

        private volatile T module;

        public NestedModuleDescriptor(ModuleFactory moduleFactory) {
            super(moduleFactory);
        }

        @Override
        public void disabled() {
            if (module instanceof StateAware) {
                ((StateAware) module).disabled();
            }
            module = null;
            super.disabled();
        }

        @Override
        public T getModule() {
            if (module == null) {
                synchronized (this) {
                    if (module == null) {
                        module = moduleFactory.createModule(getModuleClassName(), this);
                    }
                }
            }
            return module;
        }

        @Override
        protected void provideValidationRules(final ValidationPattern pattern) {
            pattern.rule(test("@class").withError("The 'class' attribute is required"));
        }
    }
}